/**
 * ChatGram - Real-time Chat JavaScript
 * Handles AJAX polling, message rendering, and user interactions
 */

const ChatGram = {
    config: {},
    lastTimestamp: 0,
    pollInterval: null,
    heartbeatInterval: null,
    activeTab: 'general',
    conversations: {},

    /**
     * Initialize chat
     */
    init(config) {
        this.config = config;
        console.log('[ChatGram] Initializing...', config);

        this.setupEventListeners();
        this.startPolling();
        this.startHeartbeat();
    },

    /**
     * Setup event listeners
     */
    setupEventListeners() {
        const sendBtn = document.getElementById('cg-send');
        const inputArea = document.getElementById('cg-input');

        // Send button
        sendBtn?.addEventListener('click', () => this.sendMessage());

        // Enter to send (Shift+Enter for new line)
        inputArea?.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.sendMessage();
            }
        });

        // Tab switching
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('cg-tab')) {
                this.switchTab(e.target.dataset.tab);
            }
        });

        // User click for private message
        document.addEventListener('click', (e) => {
            if (e.target.closest('.cg-user-item.clickable')) {
                const userItem = e.target.closest('.cg-user-item');
                const userId = userItem.dataset.userId;
                const userType = userItem.dataset.userType;
                const userName = userItem.querySelector('.cg-user-name').textContent;

                if (this.config.allowPrivateMessages) {
                    this.openPrivateChat(userId, userType, userName);
                }
            }
        });
    },

    /**
     * Start polling for messages and users
     */
    startPolling() {
        // Initial fetch
        this.fetchMessages();
        this.updateUserList();

        // Poll every 3 seconds
        this.pollInterval = setInterval(() => {
            this.fetchMessages();
            this.updateUserList();
        }, 3000);
    },

    /**
     * Start heartbeat to maintain presence
     */
    startHeartbeat() {
        // Send heartbeat every 15 seconds
        this.heartbeatInterval = setInterval(() => {
            this.sendHeartbeat();
        }, 15000);

        // Send initial heartbeat
        this.sendHeartbeat();
    },

    /**
     * Fetch new messages from server
     */
    async fetchMessages() {
        try {
            const response = await fetch(
                '/plugin_chatgram/api/messages.php?' +
                'instance_id=' + encodeURIComponent(this.config.instanceId) +
                '&since_timestamp=' + this.lastTimestamp
            );

            const data = await response.json();

            console.log('[ChatGram] Fetch messages response:', data);

            if (data.success && data.messages.length > 0) {
                this.renderMessages(data.messages);
                this.lastTimestamp = data.server_time;
            } else if (!data.success) {
                console.error('[ChatGram] Fetch messages error:', data.error);
            }

            // Remove loading message
            const loadingEls = document.querySelectorAll('.cg-loading');
            loadingEls.forEach(el => el.remove());

        } catch (error) {
            console.error('[ChatGram] Failed to fetch messages:', error);
            this.showStatus('Connection error. Retrying...', 'error');
        }
    },

    /**
     * Send a message
     */
    async sendMessage() {
        const inputArea = document.getElementById('cg-input');
        const content = inputArea.value.trim();

        if (!content) {
            return;
        }

        // Disable input while sending
        inputArea.disabled = true;
        document.getElementById('cg-send').disabled = true;

        try {
            // Check if sending to private conversation
            if (this.activeTab !== 'general' && this.conversations[this.activeTab]) {
                await this.sendPrivateMessage(this.activeTab, content);
            } else {
                // Send to general chat
                const formData = new FormData();
                formData.append('instance_id', this.config.instanceId);
                formData.append('content', content);

                const response = await fetch('/plugin_chatgram/api/post.php', {
                    method: 'POST',
                    body: formData
                });

                const data = await response.json();

                console.log('[ChatGram] Send message response:', data);

                if (data.debug) {
                    console.log('[ChatGram] Send message debug:', data.debug);
                }

                if (data.success) {
                    // Immediately fetch new messages
                    this.fetchMessages();
                } else {
                    console.error('[ChatGram] Send message error:', data.error);
                    this.showStatus(data.error || 'Failed to send message', 'error');
                    inputArea.disabled = false;
                    document.getElementById('cg-send').disabled = false;
                    inputArea.focus();
                    return;
                }
            }

            // Clear input on success
            inputArea.value = '';

        } catch (error) {
            console.error('[ChatGram] Failed to send message:', error);
            this.showStatus('Failed to send message', 'error');
        } finally {
            inputArea.disabled = false;
            document.getElementById('cg-send').disabled = false;
            inputArea.focus();
        }
    },

    /**
     * Send heartbeat to maintain presence
     */
    async sendHeartbeat() {
        try {
            const formData = new FormData();
            formData.append('instance_id', this.config.instanceId);

            const response = await fetch('/plugin_chatgram/api/heartbeat.php', {
                method: 'POST',
                body: formData
            });

            const data = await response.json();
            console.log('[ChatGram] Heartbeat response:', data);

            if (data.debug) {
                console.log('[ChatGram] Heartbeat debug:', data.debug);
            }

            if (!data.success) {
                console.error('[ChatGram] Heartbeat error:', data.error);
            }
        } catch (error) {
            console.error('[ChatGram] Heartbeat failed:', error);
        }
    },

    /**
     * Update online users list
     */
    async updateUserList() {
        try {
            const response = await fetch(
                `/plugin_chatgram/api/users.php?instance_id=${encodeURIComponent(this.config.instanceId)}`
            );

            const data = await response.json();
            console.log('[ChatGram] User list response:', data);

            if (data.success) {
                console.log('[ChatGram] Users online:', data.users);
                this.renderUserList(data.users);
                document.getElementById('cg-user-count').textContent = data.count;
            } else {
                console.error('[ChatGram] User list error:', data.error);
            }

        } catch (error) {
            console.error('[ChatGram] Failed to update user list:', error);
        }
    },

    /**
     * Render messages in the chat
     */
    renderMessages(messages) {
        const container = document.getElementById('cg-messages-general');

        messages.forEach(msg => {
            // Check if message already exists
            if (document.querySelector(`[data-message-id="${msg.id}"]`)) {
                return;
            }

            const msgEl = this.createMessageElement(msg);
            container.appendChild(msgEl);
        });

        // Auto-scroll to bottom
        container.scrollTop = container.scrollHeight;
    },

    /**
     * Create message DOM element
     */
    createMessageElement(msg) {
        const div = document.createElement('div');
        div.className = 'cg-message';
        div.dataset.messageId = msg.id;

        // Check if this is current user's message
        const isOwnMessage = (this.config.isGuest && msg.author_type === 'guest') ||
                            (!this.config.isGuest && msg.author === this.config.currentUser);

        if (isOwnMessage) {
            div.classList.add('own');
        }

        const avatarHtml = msg.avatar_url
            ? `<img src="${this.escapeHtml(msg.avatar_url)}" alt="${this.escapeHtml(msg.display_name)}">`
            : `<span class="cg-default-avatar">👤</span>`;

        div.innerHTML = `
            <div class="cg-message-avatar">
                ${avatarHtml}
            </div>
            <div class="cg-message-content">
                <div class="cg-message-header">
                    <span class="cg-message-author">${this.escapeHtml(msg.display_name)}</span>
                    ${msg.role !== 'guest' ? `<span class="cg-user-badge cg-badge-${msg.role}">${msg.role.toUpperCase()}</span>` : ''}
                    ${msg.author_type === 'guest' ? '<span class="cg-user-badge cg-badge-guest">GUEST</span>' : ''}
                    <span class="cg-message-time">${this.formatTime(msg.timestamp)}</span>
                </div>
                <div class="cg-message-text">${this.escapeHtml(msg.content)}</div>
            </div>
        `;

        return div;
    },

    /**
     * Render user list
     */
    renderUserList(users) {
        const container = document.getElementById('cg-users-list');

        container.innerHTML = '';

        if (users.length === 0) {
            container.innerHTML = '<p style="color: #666; font-size: 0.9em;">No users online</p>';
            return;
        }

        users.forEach(user => {
            const div = document.createElement('div');
            div.className = 'cg-user-item';
            div.dataset.userId = user.user_id;
            div.dataset.userType = user.user_type;

            // Don't allow private message to self
            const isSelf = (this.config.isGuest && user.user_type === 'guest') ||
                          (!this.config.isGuest && user.user_id === this.config.currentUser);

            if (!isSelf && this.config.allowPrivateMessages) {
                div.classList.add('clickable');
                div.title = 'Click to send private message';
            }

            const avatarHtml = user.avatar_url
                ? `<img class="cg-user-avatar" src="${this.escapeHtml(user.avatar_url)}" alt="${this.escapeHtml(user.display_name)}">`
                : `<span class="cg-user-avatar cg-default-avatar">👤</span>`;

            div.innerHTML = `
                ${avatarHtml}
                <div class="cg-user-info">
                    <span class="cg-user-name">${this.escapeHtml(user.display_name)}</span>
                    ${user.role !== 'guest' ? `<span class="cg-user-badge cg-badge-${user.role}">${user.role.toUpperCase()}</span>` : ''}
                    ${user.user_type === 'guest' ? '<span class="cg-user-badge cg-badge-guest">GUEST</span>' : ''}
                </div>
            `;

            container.appendChild(div);
        });
    },

    /**
     * Open private chat tab
     */
    openPrivateChat(userId, userType, userName) {
        const conversationId = this.getConversationId(
            this.config.isGuest ? this.config.currentUser : this.config.currentUser,
            this.config.isGuest ? 'guest' : 'user',
            userId,
            userType
        );

        // Check if tab already exists
        if (document.querySelector(`[data-tab="${conversationId}"]`)) {
            this.switchTab(conversationId);
            return;
        }

        // Create tab
        const tabBar = document.querySelector('.cg-tabs');
        const tab = document.createElement('div');
        tab.className = 'cg-tab';
        tab.dataset.tab = conversationId;
        tab.innerHTML = `${this.escapeHtml(userName)} <span class="cg-close-tab" title="Close">&times;</span>`;
        tabBar.appendChild(tab);

        // Handle tab close
        const closeBtn = tab.querySelector('.cg-close-tab');
        closeBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            this.closePrivateChat(conversationId);
        });

        // Create messages container
        const messagesPanel = document.querySelector('.cg-messages-panel');
        const messagesDiv = document.createElement('div');
        messagesDiv.id = `cg-messages-${conversationId}`;
        messagesDiv.className = 'cg-messages';
        messagesPanel.appendChild(messagesDiv);

        // Store conversation info
        this.conversations[conversationId] = {
            userId: userId,
            userType: userType,
            userName: userName,
            lastTimestamp: 0
        };

        // Switch to new tab
        this.switchTab(conversationId);

        // Start polling for this conversation
        this.pollPrivateMessages(conversationId);
    },

    /**
     * Close private chat tab
     */
    closePrivateChat(conversationId) {
        // Remove tab
        const tab = document.querySelector(`[data-tab="${conversationId}"]`);
        if (tab) {
            tab.remove();
        }

        // Remove messages container
        const messages = document.getElementById(`cg-messages-${conversationId}`);
        if (messages) {
            messages.remove();
        }

        // Remove from conversations
        delete this.conversations[conversationId];

        // Switch to general tab
        this.switchTab('general');
    },

    /**
     * Poll for private messages in a conversation
     */
    async pollPrivateMessages(conversationId) {
        if (!this.conversations[conversationId]) {
            return; // Conversation closed
        }

        try {
            const conv = this.conversations[conversationId];
            const response = await fetch(
                `/plugin_chatgram/api/private-message.php?` +
                `action=get&instance_id=${encodeURIComponent(this.config.instanceId)}&` +
                `conversation_id=${encodeURIComponent(conversationId)}&` +
                `since_timestamp=${conv.lastTimestamp}`
            );

            const data = await response.json();

            if (data.success && data.messages.length > 0) {
                this.renderPrivateMessages(conversationId, data.messages);
                conv.lastTimestamp = data.server_time;
            }

            // Update unread indicator
            if (data.unread_count > 0 && this.activeTab !== conversationId) {
                const tab = document.querySelector(`[data-tab="${conversationId}"]`);
                if (tab && !tab.classList.contains('unread')) {
                    tab.classList.add('unread');
                }
            }

            // Continue polling if conversation still open
            if (this.conversations[conversationId]) {
                setTimeout(() => this.pollPrivateMessages(conversationId), 3000);
            }

        } catch (error) {
            console.error('[ChatGram] Failed to poll private messages:', error);
            // Retry after delay if conversation still open
            if (this.conversations[conversationId]) {
                setTimeout(() => this.pollPrivateMessages(conversationId), 5000);
            }
        }
    },

    /**
     * Render private messages
     */
    renderPrivateMessages(conversationId, messages) {
        const container = document.getElementById(`cg-messages-${conversationId}`);
        if (!container) return;

        messages.forEach(msg => {
            // Check if message already exists
            if (container.querySelector(`[data-message-id="${msg.id}"]`)) {
                return;
            }

            const msgEl = this.createPrivateMessageElement(msg);
            container.appendChild(msgEl);
        });

        // Auto-scroll to bottom if this is the active tab
        if (this.activeTab === conversationId) {
            container.scrollTop = container.scrollHeight;
        }
    },

    /**
     * Create private message DOM element
     */
    createPrivateMessageElement(msg) {
        const div = document.createElement('div');
        div.className = 'cg-message';
        div.dataset.messageId = msg.id;

        // Check if this is current user's message
        const currentId = this.config.isGuest ? 'guest' : 'user';
        const isOwnMessage = (msg.from_type === currentId &&
                             msg.from_user === this.config.currentUser);

        if (isOwnMessage) {
            div.classList.add('own');
        }

        div.innerHTML = `
            <div class="cg-message-content">
                <div class="cg-message-header">
                    <span class="cg-message-author">${isOwnMessage ? 'You' : 'Them'}</span>
                    <span class="cg-message-time">${this.formatTime(msg.timestamp)}</span>
                </div>
                <div class="cg-message-text">${this.escapeHtml(msg.content)}</div>
            </div>
        `;

        return div;
    },

    /**
     * Send private message
     */
    async sendPrivateMessage(conversationId, content) {
        const conv = this.conversations[conversationId];
        if (!conv) return;

        try {
            const formData = new FormData();
            formData.append('action', 'send');
            formData.append('instance_id', this.config.instanceId);
            formData.append('to_user', conv.userId);
            formData.append('to_type', conv.userType);
            formData.append('content', content);

            const response = await fetch('/plugin_chatgram/api/private-message.php', {
                method: 'POST',
                body: formData
            });

            const data = await response.json();

            if (data.success) {
                // Immediately poll for new messages
                this.pollPrivateMessages(conversationId);
            } else {
                this.showStatus(data.error || 'Failed to send private message', 'error');
            }

        } catch (error) {
            console.error('[ChatGram] Failed to send private message:', error);
            this.showStatus('Failed to send private message', 'error');
        }
    },

    /**
     * Switch active tab
     */
    switchTab(tabName) {
        // Update tabs
        document.querySelectorAll('.cg-tab').forEach(tab => {
            tab.classList.remove('active');
            if (tab.dataset.tab === tabName) {
                tab.classList.add('active');
                // Clear unread indicator
                tab.classList.remove('unread');
            }
        });

        // Update messages visibility
        document.querySelectorAll('.cg-messages').forEach(msgs => {
            msgs.classList.remove('active');
        });

        const activeMessages = document.getElementById(`cg-messages-${tabName}`);
        if (activeMessages) {
            activeMessages.classList.add('active');
            // Auto-scroll to bottom
            activeMessages.scrollTop = activeMessages.scrollHeight;
        }

        this.activeTab = tabName;

        // Focus input
        document.getElementById('cg-input')?.focus();
    },

    /**
     * Get conversation ID (alphabetically sorted for consistency)
     */
    getConversationId(user1, type1, user2, type2) {
        const id1 = `${type1}_${user1}`;
        const id2 = `${type2}_${user2}`;
        const ids = [id1, id2].sort();
        return `conv_${ids[0]}_${ids[1]}`;
    },

    /**
     * Show status message
     */
    showStatus(message, type = 'info') {
        const statusEl = document.getElementById('cg-status');
        if (!statusEl) return;

        statusEl.textContent = message;
        statusEl.className = 'cg-status cg-status-' + type;
        statusEl.style.display = 'block';

        setTimeout(() => {
            statusEl.style.display = 'none';
        }, 5000);
    },

    /**
     * Escape HTML
     */
    escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    },

    /**
     * Format timestamp
     */
    formatTime(timestamp) {
        const date = new Date(timestamp * 1000);
        return date.toLocaleTimeString('en-US', {
            hour: 'numeric',
            minute: '2-digit'
        });
    },

    /**
     * Cleanup on page unload
     */
    destroy() {
        if (this.pollInterval) {
            clearInterval(this.pollInterval);
        }
        if (this.heartbeatInterval) {
            clearInterval(this.heartbeatInterval);
        }
    }
};

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    ChatGram.destroy();
});
