<?php
/**
 * ChatGram Embed Handler
 * Renders chat interface inline in article
 */

// Get instance context
$instance_path = $_ENV['BOOKGRAM_INSTANCE_PATH'] ?? null;
$instance_id = $_ENV['BOOKGRAM_INSTANCE_ID'] ?? null;
$page_uuid = $_ENV['BOOKGRAM_PAGE_UUID'] ?? null;

if (!$instance_path || !$instance_id) {
    echo '<p style="color: #cc0000;">Error: ChatGram instance context not available.</p>';
    return;
}

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';

// Check if configured
if (!file_exists($config_file)) {
    echo '<div style="padding: 20px; background: #fff3cd; border: 2px solid #ff6600; margin: 20px 0;">';
    echo '<h3>Chat Not Configured</h3>';
    echo '<p><a href="#" onclick="event.preventDefault(); openPluginAdmin(\'' .
         htmlspecialchars($page_uuid) . '\', \'plugin_chatgram\', \'' .
         htmlspecialchars($instance_id) . '\', \'ChatGram\');">Configure Chat Room</a></p>';
    echo '</div>';
    return;
}

// Load config
$config = json_decode(file_get_contents($config_file), true);

// Load BookGram auth functions if available
if (file_exists(dirname(__DIR__) . '/app/core/functions.php')) {
    require_once dirname(__DIR__) . '/app/core/functions.php';
}

// Check if user is logged in
$is_logged_in = function_exists('is_logged_in') ? is_logged_in() : false;
$current_user = 'Anonymous';
$current_role = 'guest';
$avatar_url = null;

if ($is_logged_in) {
    $current_user = get_current_username() ?? 'User';
    $current_role = is_admin() ? 'admin' : (is_editor() ? 'editor' : 'viewer');

    // Get avatar if available
    $users = load_users();
    if (isset($users[$current_user]) && !empty($users[$current_user]['photo_url'])) {
        $avatar_url = $users[$current_user]['photo_url'];
    }
}

// Check access permissions
$can_access = false;

if ($config['access_mode'] === 'login_only' && $is_logged_in) {
    $can_access = true;
} elseif ($config['access_mode'] === 'login_and_guest') {
    $can_access = true;
}

// Load system config to check if ChatGram is enabled
$system_config_file = dirname(__DIR__) . '/app/storage/config.json';
$system_config = file_exists($system_config_file) ? json_decode(file_get_contents($system_config_file), true) : [];
$chatgram_enabled = $system_config['chatgram']['enabled'] ?? false;

if (!$chatgram_enabled) {
    echo '<div style="padding: 15px; background: #f0f0f0; border: 1px solid #808080; margin: 20px 0;">';
    echo '<p><strong>Chat is currently disabled system-wide.</strong></p>';
    echo '</div>';
    return;
}

if (!$can_access && $config['access_mode'] === 'login_only') {
    echo '<div style="padding: 15px; background: #f0f0f0; border: 1px solid #808080; margin: 20px 0;">';
    echo '<p><strong>Login required to access this chat.</strong> <a href="/login">Login to BookGram</a></p>';
    echo '</div>';
    return;
}

// For guests, check if they have a session
$is_guest = !$is_logged_in;
$needs_guest_setup = false;

if ($is_guest && $config['access_mode'] === 'login_and_guest') {
    require_once dirname(__DIR__) . '/plugin_chatgram/lib/GuestManager.php';
    $guest_info = GuestManager::getGuestInfo();

    if (!$guest_info || !isset($_SESSION['chatgram_instance_id']) || $_SESSION['chatgram_instance_id'] !== $instance_id) {
        $needs_guest_setup = true;
    }
}

?>

<link rel="stylesheet" href="/plugin_chatgram/css/chatgram.css">

<div class="chatgram-container" id="chatgram-<?php echo htmlspecialchars($instance_id); ?>"
     data-instance-id="<?php echo htmlspecialchars($instance_id); ?>"
     data-page-uuid="<?php echo htmlspecialchars($page_uuid); ?>">

    <?php if ($needs_guest_setup): ?>
        <!-- Guest Name Prompt -->
        <div id="cg-guest-prompt" class="cg-guest-prompt">
            <h3>Join <?php echo htmlspecialchars($config['chat_name'] ?? 'Chat Room'); ?></h3>
            <?php if (!empty($config['welcome_message'])): ?>
                <p><?php echo nl2br(htmlspecialchars($config['welcome_message'])); ?></p>
            <?php endif; ?>
            <p>Enter a display name to join the conversation:</p>
            <input type="text" id="cg-guest-name" placeholder="Your name" maxlength="30" autocomplete="off">
            <div id="cg-guest-error" style="color: #cc0000; margin: 10px 0; display: none;"></div>
            <button id="cg-guest-join">Join Chat</button>
            <p class="cg-note" style="font-size: 0.9em; color: #666;">
                <a href="/login">Login to BookGram</a> for full features and persistent identity.
            </p>
        </div>

    <?php else: ?>
        <!-- Chat Interface -->
        <div class="cg-header">
            <h3><?php echo htmlspecialchars($config['chat_name'] ?? 'Chat Room'); ?></h3>
            <div class="cg-online-count">
                Online: <span id="cg-user-count">0</span>
            </div>
        </div>

        <!-- Tab Bar -->
        <div class="cg-tabs">
            <div class="cg-tab active" data-tab="general">General</div>
            <!-- Private chat tabs will be added here dynamically -->
        </div>

        <!-- Content Area -->
        <div class="cg-content">
            <!-- Left: Messages Panel -->
            <div class="cg-messages-panel">
                <div id="cg-messages-general" class="cg-messages active">
                    <div class="cg-loading">Loading messages...</div>
                </div>
                <!-- Private message containers will be added here dynamically -->
            </div>

            <!-- Right: User List -->
            <div class="cg-users-panel">
                <h4>Online Users</h4>
                <div id="cg-users-list">
                    <div class="cg-loading">Loading...</div>
                </div>
            </div>
        </div>

        <!-- Input Area -->
        <div class="cg-input-area">
            <textarea id="cg-input" placeholder="Type a message..." rows="2"></textarea>
            <button id="cg-send">Send</button>
        </div>

        <div class="cg-status" id="cg-status"></div>
    <?php endif; ?>
</div>

<script src="/plugin_chatgram/js/chatgram.js"></script>
<script>
// Initialize ChatGram
const chatgramConfig = {
    instanceId: '<?php echo addslashes($instance_id); ?>',
    pageUuid: '<?php echo addslashes($page_uuid); ?>',
    currentUser: '<?php echo addslashes($current_user); ?>',
    currentRole: '<?php echo addslashes($current_role); ?>',
    avatarUrl: <?php echo $avatar_url ? "'" . addslashes($avatar_url) . "'" : 'null'; ?>,
    isGuest: <?php echo $is_guest ? 'true' : 'false'; ?>,
    allowPrivateMessages: <?php echo ($config['allow_private_messages'] ?? true) ? 'true' : 'false'; ?>,
    needsGuestSetup: <?php echo $needs_guest_setup ? 'true' : 'false'; ?>
};

// Only initialize if not in guest setup mode
if (!chatgramConfig.needsGuestSetup) {
    ChatGram.init(chatgramConfig);
}

// Handle guest setup
<?php if ($needs_guest_setup): ?>
document.getElementById('cg-guest-join')?.addEventListener('click', function() {
    const nameInput = document.getElementById('cg-guest-name');
    const errorDiv = document.getElementById('cg-guest-error');
    const guestName = nameInput.value.trim();

    if (!guestName) {
        errorDiv.textContent = 'Please enter a name.';
        errorDiv.style.display = 'block';
        return;
    }

    // Send guest setup request
    fetch('/plugin_chatgram/api/guest-setup.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: 'guest_name=' + encodeURIComponent(guestName) +
              '&instance_id=' + encodeURIComponent(chatgramConfig.instanceId)
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            // Reload page to start chat
            window.location.reload();
        } else {
            errorDiv.textContent = data.error || 'Invalid name. Please try another.';
            errorDiv.style.display = 'block';
        }
    })
    .catch(err => {
        errorDiv.textContent = 'Error setting up guest session. Please try again.';
        errorDiv.style.display = 'block';
    });
});

// Allow Enter key to join
document.getElementById('cg-guest-name')?.addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        document.getElementById('cg-guest-join').click();
    }
});
<?php endif; ?>
</script>
