<?php
/**
 * ChatGram API - Private Messages
 * Handles sending and retrieving private messages
 */

session_start();
header('Content-Type: application/json');

// Get parameters
$action = $_POST['action'] ?? $_GET['action'] ?? null;
$instance_id = $_POST['instance_id'] ?? $_GET['instance_id'] ?? null;

if (!$action || !$instance_id) {
    echo json_encode(['success' => false, 'error' => 'Missing required parameters']);
    exit;
}

// Find instance path
function find_instance_path($instance_id) {
    $base = dirname(__DIR__, 2) . '/app/storage/plugin_instances/';
    if (!is_dir($base)) {
        return null;
    }
    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_chatgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }
    return null;
}

$instance_path = find_instance_path($instance_id);
if (!$instance_path) {
    echo json_encode(['success' => false, 'error' => 'Instance not found']);
    exit;
}

// Load libraries
require_once dirname(__DIR__) . '/lib/ChatStorage.php';
require_once dirname(__DIR__) . '/lib/GuestManager.php';
require_once dirname(__DIR__, 2) . '/app/core/functions.php';

$storage = new ChatStorage($instance_path);
$config = $storage->loadConfig();

// Get current user info
$is_logged_in = is_logged_in();
$current_user_id = null;
$current_user_type = 'guest';

if ($is_logged_in) {
    $current_user_id = get_current_username();
    $current_user_type = 'user';
} else {
    $guest_info = GuestManager::getGuestInfo();
    if (!$guest_info) {
        echo json_encode(['success' => false, 'error' => 'Not authenticated']);
        exit;
    }
    $current_user_id = $guest_info['guest_id'];
}

/**
 * Generate conversation ID
 */
function getConversationId($user1, $type1, $user2, $type2) {
    $id1 = $type1 . '_' . $user1;
    $id2 = $type2 . '_' . $user2;
    $ids = [$id1, $id2];
    sort($ids);
    return 'conv_' . implode('_', $ids);
}

// === SEND PRIVATE MESSAGE ===
if ($action === 'send') {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'error' => 'Method not allowed']);
        exit;
    }

    $to_user = $_POST['to_user'] ?? null;
    $to_type = $_POST['to_type'] ?? null;
    $content = trim($_POST['content'] ?? '');

    if (!$to_user || !$to_type || empty($content)) {
        echo json_encode(['success' => false, 'error' => 'Missing required parameters']);
        exit;
    }

    // Sanitize content
    $content = strip_tags($content);
    $content = htmlspecialchars($content, ENT_QUOTES, 'UTF-8');
    $content = substr($content, 0, 1000);

    // Create conversation ID
    $conversation_id = getConversationId($current_user_id, $current_user_type, $to_user, $to_type);

    // Add private message
    $message = $storage->addPrivateMessage(
        $conversation_id,
        $current_user_id,
        $current_user_type,
        $to_user,
        $to_type,
        $content
    );

    if ($message) {
        echo json_encode([
            'success' => true,
            'message' => $message
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Failed to save message']);
    }
    exit;
}

// === GET PRIVATE MESSAGES ===
if ($action === 'get') {
    $conversation_id = $_GET['conversation_id'] ?? null;
    $since_timestamp = intval($_GET['since_timestamp'] ?? 0);

    if (!$conversation_id) {
        echo json_encode(['success' => false, 'error' => 'Missing conversation_id']);
        exit;
    }

    // Verify user is participant in conversation
    $parts = explode('_', $conversation_id);
    if (count($parts) !== 5 || $parts[0] !== 'conv') {
        echo json_encode(['success' => false, 'error' => 'Invalid conversation_id']);
        exit;
    }

    $participant1 = $parts[1] . '_' . $parts[2];
    $participant2 = $parts[3] . '_' . $parts[4];
    $current_id = $current_user_type . '_' . $current_user_id;

    if ($current_id !== $participant1 && $current_id !== $participant2) {
        echo json_encode(['success' => false, 'error' => 'Access denied']);
        exit;
    }

    // Get messages
    $messages = $storage->getPrivateMessagesSince($conversation_id, $since_timestamp);

    // Mark messages as read
    $storage->markMessagesAsRead($conversation_id, $current_user_id, $current_user_type);

    // Get unread count
    $unread_count = $storage->getUnreadCount($conversation_id, $current_user_id, $current_user_type);

    echo json_encode([
        'success' => true,
        'messages' => $messages,
        'unread_count' => $unread_count,
        'server_time' => time()
    ]);
    exit;
}

echo json_encode(['success' => false, 'error' => 'Invalid action']);
