<?php
/**
 * ChatGram API - Post Message
 * Handles posting new public messages
 */

// Suppress errors and warnings to prevent breaking JSON output
error_reporting(0);
ini_set('display_errors', 0);

// Start output buffering to prevent any stray output
ob_start();

session_start();
header('Content-Type: application/json');

// Wrap everything in try-catch to ensure JSON output
try {

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Get parameters
$instance_id = $_POST['instance_id'] ?? null;
$content = trim($_POST['content'] ?? '');

if (!$instance_id || empty($content)) {
    echo json_encode(['success' => false, 'error' => 'Missing required parameters']);
    exit;
}

// Find instance path
function find_instance_path($instance_id) {
    $base = dirname(__DIR__, 2) . '/app/storage/plugin_instances/';

    if (!is_dir($base)) {
        return null;
    }

    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_chatgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }

    return null;
}

$instance_path = find_instance_path($instance_id);

if (!$instance_path) {
    echo json_encode(['success' => false, 'error' => 'Instance not found']);
    exit;
}

// Load libraries
require_once dirname(__DIR__) . '/lib/ChatStorage.php';
require_once dirname(__DIR__) . '/lib/GuestManager.php';
require_once dirname(__DIR__) . '/lib/RateLimiter.php';

// Try to load core functions
$functions_file = dirname(__DIR__, 2) . '/app/core/functions.php';
if (file_exists($functions_file)) {
    require_once $functions_file;
}

$storage = new ChatStorage($instance_path);
$config = $storage->loadConfig();

// Load system config for rate limits
$system_config_file = dirname(__DIR__, 2) . '/app/storage/config.json';
$system_config = file_exists($system_config_file) ? json_decode(file_get_contents($system_config_file), true) : [];
$rate_limit = $system_config['chatgram']['rate_limit_messages_per_minute'] ?? 10;
$max_length = $system_config['chatgram']['max_message_length'] ?? 1000;

// Check if user is logged in
$is_logged_in = function_exists('is_logged_in') ? is_logged_in() : false;

// Fallback: Check session directly if function doesn't exist or returns false
if (!$is_logged_in && isset($_SESSION['bookgram_user'])) {
    $is_logged_in = true;
}

$user_id = null;
$user_type = 'guest';
$display_name = 'Anonymous';
$avatar_url = null;
$role = 'guest';

if ($is_logged_in) {
    // Logged-in user
    $user_id = function_exists('get_current_username') ? get_current_username() : null;

    // Fallback: Get username from session
    if (!$user_id && isset($_SESSION['bookgram_user'])) {
        $user_id = $_SESSION['bookgram_user'];
    }

    $user_type = 'user';
    $display_name = $user_id ?: 'Unknown';

    if (function_exists('is_admin') && function_exists('is_editor')) {
        $role = is_admin() ? 'admin' : (is_editor() ? 'editor' : 'viewer');
    } else if (isset($_SESSION['bookgram_role'])) {
        // Fallback: Get role from session
        $role = $_SESSION['bookgram_role'];
    }

    // Get avatar if available
    if (function_exists('load_users')) {
        $users = load_users();
        if (isset($users[$user_id]) && !empty($users[$user_id]['photo_url'])) {
            $avatar_url = $users[$user_id]['photo_url'];
        }
    }
} else {
    // Guest user
    if ($config['access_mode'] !== 'login_and_guest') {
        echo json_encode(['success' => false, 'error' => 'Login required']);
        exit;
    }

    $guest_info = GuestManager::getGuestInfo();
    if (!$guest_info) {
        echo json_encode(['success' => false, 'error' => 'Guest session not found']);
        exit;
    }

    $user_id = $guest_info['guest_id'];
    $display_name = $guest_info['display_name'];
}

// Sanitize content
$content = strip_tags($content);
$content = htmlspecialchars($content, ENT_QUOTES, 'UTF-8');
$content = substr($content, 0, $max_length);

if (empty($content)) {
    echo json_encode(['success' => false, 'error' => 'Message cannot be empty']);
    exit;
}

// Check rate limit
$rate_limiter = new RateLimiter($storage, $rate_limit);
if (!$rate_limiter->checkLimit($user_id)) {
    $reset_time = $rate_limiter->getResetTime($user_id);
    echo json_encode([
        'success' => false,
        'error' => 'Rate limit exceeded. Please wait before posting again.',
        'retry_after' => $reset_time
    ]);
    exit;
}

// Add message
$message = $storage->addMessage(
    $user_id,
    $user_type,
    $display_name,
    $avatar_url,
    $role,
    $content
);

if ($message) {
    // Trigger cleanup (1% chance)
    if (rand(1, 100) === 1) {
        cleanup_old_messages($storage, $system_config);
    }

    // Clear any buffered output and send JSON
    ob_clean();

    echo json_encode([
        'success' => true,
        'message' => $message,
        'debug' => [
            'user_id' => $user_id,
            'user_type' => $user_type,
            'role' => $role,
            'is_logged_in' => $is_logged_in,
            'session_user' => $_SESSION['bookgram_user'] ?? 'not set'
        ]
    ]);
} else {
    ob_clean();
    echo json_encode(['success' => false, 'error' => 'Failed to save message']);
}

} catch (Exception $e) {
    // Clear any buffered output
    ob_clean();

    echo json_encode([
        'success' => false,
        'error' => 'Server error: ' . $e->getMessage()
    ]);
}

// Flush output buffer
ob_end_flush();

/**
 * Cleanup old messages based on retention period
 */
function cleanup_old_messages($storage, $system_config) {
    $retention_days = $system_config['chatgram']['message_retention_days'] ?? 30;
    $cutoff = time() - ($retention_days * 86400);

    $messages = $storage->loadMessages();
    $original_count = count($messages);

    $messages = array_filter($messages, function($msg) use ($cutoff) {
        return $msg['timestamp'] >= $cutoff;
    });

    if (count($messages) < $original_count) {
        $storage->saveMessages(array_values($messages));
    }
}
