<?php
/**
 * ChatGram API - Heartbeat
 * Updates user presence to keep them in online list
 */

// Suppress errors and warnings to prevent breaking JSON output
error_reporting(0);
ini_set('display_errors', 0);

// Start output buffering to prevent any stray output
ob_start();

session_start();
header('Content-Type: application/json');

// Wrap everything in try-catch to ensure JSON output
try {

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Get parameters
$instance_id = $_POST['instance_id'] ?? null;

if (!$instance_id) {
    echo json_encode(['success' => false, 'error' => 'Missing instance_id']);
    exit;
}

// Find instance path
function find_instance_path($instance_id) {
    $base = dirname(__DIR__, 2) . '/app/storage/plugin_instances/';

    if (!is_dir($base)) {
        return null;
    }

    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_chatgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }

    return null;
}

$instance_path = find_instance_path($instance_id);

if (!$instance_path) {
    echo json_encode(['success' => false, 'error' => 'Instance not found']);
    exit;
}

// Load libraries
require_once dirname(__DIR__) . '/lib/ChatStorage.php';
require_once dirname(__DIR__) . '/lib/GuestManager.php';

// Try to load core functions
$functions_file = dirname(__DIR__, 2) . '/app/core/functions.php';
if (file_exists($functions_file)) {
    require_once $functions_file;
}

$storage = new ChatStorage($instance_path);
$config = $storage->loadConfig();

// Check if user is logged in
$is_logged_in = function_exists('is_logged_in') ? is_logged_in() : false;

// Fallback: Check session directly if function doesn't exist or returns false
if (!$is_logged_in && isset($_SESSION['bookgram_user'])) {
    $is_logged_in = true;
}

$user_id = null;
$user_type = 'guest';
$display_name = 'Anonymous';
$avatar_url = null;
$role = 'guest';

// DEBUG: Log authentication status
error_log('[ChatGram Heartbeat] is_logged_in: ' . ($is_logged_in ? 'true' : 'false'));
error_log('[ChatGram Heartbeat] SESSION bookgram_user: ' . ($_SESSION['bookgram_user'] ?? 'not set'));

if ($is_logged_in) {
    // Logged-in user
    $user_id = function_exists('get_current_username') ? get_current_username() : null;

    // Fallback: Get username from session
    if (!$user_id && isset($_SESSION['bookgram_user'])) {
        $user_id = $_SESSION['bookgram_user'];
    }

    $user_type = 'user';
    $display_name = $user_id ?: 'Unknown';

    if (function_exists('is_admin') && function_exists('is_editor')) {
        $role = is_admin() ? 'admin' : (is_editor() ? 'editor' : 'viewer');
    } else if (isset($_SESSION['bookgram_role'])) {
        // Fallback: Get role from session
        $role = $_SESSION['bookgram_role'];
    }

    // Get avatar if available
    if (function_exists('load_users')) {
        $users = load_users();
        if (isset($users[$user_id]) && !empty($users[$user_id]['photo_url'])) {
            $avatar_url = $users[$user_id]['photo_url'];
        }
    }

    // DEBUG: Log logged-in user details
    error_log('[ChatGram Heartbeat] Logged-in user - ID: ' . $user_id . ', Type: ' . $user_type . ', Role: ' . $role . ', Name: ' . $display_name);
} else {
    // Guest user
    if ($config['access_mode'] !== 'login_and_guest') {
        echo json_encode(['success' => false, 'error' => 'Login required']);
        exit;
    }

    $guest_info = GuestManager::getGuestInfo();
    if (!$guest_info) {
        echo json_encode(['success' => false, 'error' => 'Guest session not found']);
        exit;
    }

    $user_id = $guest_info['guest_id'];
    $display_name = $guest_info['display_name'];

    // Update guest activity
    $storage->updateGuestActivity($user_id);
}

// Update user heartbeat
error_log('[ChatGram Heartbeat] Updating heartbeat for: ' . $user_id . ' (type: ' . $user_type . ')');
$storage->updateUserHeartbeat($user_id, $user_type, $display_name, $avatar_url, $role);
error_log('[ChatGram Heartbeat] Heartbeat updated successfully');

// Clean up inactive guests (10% chance)
if (!$is_logged_in && rand(1, 10) === 1) {
    $system_config_file = dirname(__DIR__, 2) . '/app/storage/config.json';
    $system_config = file_exists($system_config_file) ? json_decode(file_get_contents($system_config_file), true) : [];
    $timeout = $system_config['chatgram']['guest_timeout_minutes'] ?? 10;

    $storage->cleanupInactiveGuests($timeout);
}

// Clear any buffered output and send JSON
ob_clean();

// Return response with debug info
echo json_encode([
    'success' => true,
    'user_id' => $user_id,
    'user_type' => $user_type,
    'display_name' => $display_name,
    'role' => $role,
    'timestamp' => time(),
    'debug' => [
        'is_logged_in' => $is_logged_in,
        'is_logged_in_exists' => function_exists('is_logged_in'),
        'get_current_username_exists' => function_exists('get_current_username'),
        'functions_file_exists' => file_exists(dirname(__DIR__, 2) . '/app/core/functions.php'),
        'session_user' => $_SESSION['bookgram_user'] ?? 'not set'
    ]
]);

} catch (Exception $e) {
    // Clear any buffered output
    ob_clean();

    echo json_encode([
        'success' => false,
        'error' => 'Server error: ' . $e->getMessage()
    ]);
}

// Flush output buffer
ob_end_flush();
