<?php
/**
 * ChatGram API - Conversations
 * Returns list of active private conversations for current user
 */

session_start();
header('Content-Type: application/json');

// Get parameters
$instance_id = $_GET['instance_id'] ?? null;

if (!$instance_id) {
    echo json_encode(['success' => false, 'error' => 'Missing instance_id']);
    exit;
}

// Find instance path
function find_instance_path($instance_id) {
    $base = dirname(__DIR__, 2) . '/app/storage/plugin_instances/';
    if (!is_dir($base)) {
        return null;
    }
    $page_dirs = glob($base . '*', GLOB_ONLYDIR);
    foreach ($page_dirs as $page_dir) {
        $plugin_path = $page_dir . '/plugin_chatgram/' . $instance_id;
        if (is_dir($plugin_path)) {
            return $plugin_path;
        }
    }
    return null;
}

$instance_path = find_instance_path($instance_id);
if (!$instance_path) {
    echo json_encode(['success' => false, 'error' => 'Instance not found']);
    exit;
}

// Load libraries
require_once dirname(__DIR__) . '/lib/ChatStorage.php';
require_once dirname(__DIR__) . '/lib/GuestManager.php';
require_once dirname(__DIR__, 2) . '/app/core/functions.php';

$storage = new ChatStorage($instance_path);

// Get current user info
$is_logged_in = is_logged_in();
$current_user_id = null;
$current_user_type = 'guest';

if ($is_logged_in) {
    $current_user_id = get_current_username();
    $current_user_type = 'user';
} else {
    $guest_info = GuestManager::getGuestInfo();
    if (!$guest_info) {
        echo json_encode(['success' => false, 'error' => 'Not authenticated']);
        exit;
    }
    $current_user_id = $guest_info['guest_id'];
}

// Load all private messages
$all_messages = $storage->loadPrivateMessages();

// Group by conversation
$conversations = [];
$current_id = $current_user_type . '_' . $current_user_id;

foreach ($all_messages as $msg) {
    $conversation_id = $msg['conversation_id'];

    // Check if user is participant
    $from_id = $msg['from_type'] . '_' . $msg['from_user'];
    $to_id = $msg['to_type'] . '_' . $msg['to_user'];

    if ($from_id !== $current_id && $to_id !== $current_id) {
        continue; // Not a participant
    }

    // Initialize conversation if not exists
    if (!isset($conversations[$conversation_id])) {
        // Determine other user
        $other_user_id = ($from_id === $current_id) ? $msg['to_user'] : $msg['from_user'];
        $other_user_type = ($from_id === $current_id) ? $msg['to_type'] : $msg['from_type'];

        // Get other user's display name
        $other_display_name = 'Unknown';
        if ($other_user_type === 'user') {
            $users = load_users();
            $other_display_name = $other_user_id;
        } else {
            $guest_session = $storage->getGuestSession($other_user_id);
            if ($guest_session) {
                $other_display_name = $guest_session['display_name'];
            }
        }

        $conversations[$conversation_id] = [
            'conversation_id' => $conversation_id,
            'other_user' => [
                'user_id' => $other_user_id,
                'user_type' => $other_user_type,
                'display_name' => $other_display_name
            ],
            'last_message' => [
                'content' => $msg['content'],
                'timestamp' => $msg['timestamp'],
                'from_me' => ($from_id === $current_id)
            ],
            'unread_count' => 0
        ];
    } else {
        // Update last message if this is newer
        if ($msg['timestamp'] > $conversations[$conversation_id]['last_message']['timestamp']) {
            $conversations[$conversation_id]['last_message'] = [
                'content' => $msg['content'],
                'timestamp' => $msg['timestamp'],
                'from_me' => ($from_id === $current_id)
            ];
        }
    }

    // Count unread messages
    if ($to_id === $current_id && !$msg['read']) {
        $conversations[$conversation_id]['unread_count']++;
    }
}

// Convert to array and sort by last message timestamp
$conversations_array = array_values($conversations);
usort($conversations_array, function($a, $b) {
    return $b['last_message']['timestamp'] - $a['last_message']['timestamp'];
});

echo json_encode([
    'success' => true,
    'conversations' => $conversations_array
]);
