<?php
/**
 * ChatGram Admin Panel
 * Manages instance settings, moderation, and statistics
 */

session_start();
require_once dirname(__DIR__, 2) . '/app/core/plugin_auth.php';

// Authenticate
$auth = bookgram_auth_check();
bookgram_require_auth();

$username = bookgram_get_username();
$is_admin = bookgram_is_admin();
$instance_path = bookgram_get_instance_path();
$instance_id = $_SESSION['bookgram_instance_id'] ?? '';

// Load storage library
require_once dirname(__DIR__) . '/lib/ChatStorage.php';
$storage = new ChatStorage($instance_path);

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';

// Load current config
$config = $storage->loadConfig();

$success_message = '';
$error_message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'save_settings') {
        // Update config
        $config['chat_name'] = trim($_POST['chat_name']) ?: 'Chat Room';
        $config['welcome_message'] = trim($_POST['welcome_message']) ?: 'Welcome!';
        $config['access_mode'] = $_POST['access_mode'] ?? 'login_only';
        $config['allow_private_messages'] = isset($_POST['allow_private_messages']);

        if ($storage->saveConfig($config)) {
            $success_message = 'Settings saved successfully!';
        } else {
            $error_message = 'Failed to save settings.';
        }
    }

    if ($action === 'delete_message' && $is_admin) {
        $message_id = $_POST['message_id'] ?? '';
        if ($storage->deleteMessage($message_id)) {
            $success_message = 'Message deleted.';
        }
    }

    if ($action === 'clear_all_messages' && $is_admin) {
        if ($storage->saveMessages([])) {
            $success_message = 'All messages cleared.';
        }
    }
}

// Load statistics
$messages = $storage->loadMessages();
$online_users = $storage->loadOnlineUsers();
$message_count = count($messages);
$online_count = count($online_users);

// Load core functions if not already loaded
if (!function_exists('e')) {
    require_once dirname(__DIR__, 3) . '/app/core/functions.php';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ChatGram Admin - <?php echo e($config['chat_name'] ?? 'Chat Room'); ?></title>
    <link rel="stylesheet" href="/public/css/style.css">
    <style>
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .stat-card {
            background: #ffffff;
            border: 2px solid #808080;
            padding: 15px;
            text-align: center;
        }
        .stat-number {
            font-size: 2em;
            font-weight: bold;
            color: #000080;
        }
        .stat-label {
            color: #666;
            margin-top: 5px;
        }
        .success {
            padding: 10px;
            background: #90ee90;
            border: 2px solid #008000;
            margin-bottom: 15px;
        }
        .error {
            padding: 10px;
            background: #ffcccc;
            border: 2px solid #cc0000;
            margin-bottom: 15px;
        }
        .message-item {
            background: #f0f0f0;
            border: 1px solid #ccc;
            padding: 10px;
            margin-bottom: 10px;
        }
        .message-header {
            font-weight: bold;
            margin-bottom: 5px;
        }
        .message-meta {
            font-size: 0.85em;
            color: #666;
        }
        .inline-form {
            display: inline;
        }
    </style>
</head>
<body style="background: #008080; padding: 20px;">
    <div class="window" style="max-width: 900px; margin: 0 auto;">
        <div class="window-title-bar">
            <span class="title">ChatGram Admin - <?php echo e($config['chat_name'] ?? 'Chat Room'); ?></span>
        </div>
        <div class="window-content">

            <?php if ($success_message): ?>
                <div class="success"><?php echo e($success_message); ?></div>
            <?php endif; ?>

            <?php if ($error_message): ?>
                <div class="error"><?php echo e($error_message); ?></div>
            <?php endif; ?>

            <!-- Statistics -->
            <h2>Statistics</h2>
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?php echo $message_count; ?></div>
                    <div class="stat-label">Total Messages</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo $online_count; ?></div>
                    <div class="stat-label">Online Now</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo e($config['access_mode'] ?? 'login_only'); ?></div>
                    <div class="stat-label">Access Mode</div>
                </div>
            </div>

            <hr>

            <!-- Settings -->
            <h2>Chat Room Settings</h2>
            <form method="POST">
                <input type="hidden" name="action" value="save_settings">

                <div style="margin-bottom: 15px;">
                    <label for="chat_name"><strong>Chat Room Name:</strong></label><br>
                    <input type="text" id="chat_name" name="chat_name" value="<?php echo e($config['chat_name'] ?? 'Chat Room'); ?>" required style="width: 100%; max-width: 400px;">
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="welcome_message"><strong>Welcome Message:</strong></label><br>
                    <textarea id="welcome_message" name="welcome_message" rows="3" style="width: 100%; max-width: 400px;"><?php echo e($config['welcome_message'] ?? 'Welcome!'); ?></textarea>
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="access_mode"><strong>Access Mode:</strong></label><br>
                    <select id="access_mode" name="access_mode" style="width: 100%; max-width: 400px;">
                        <option value="login_only" <?php echo ($config['access_mode'] ?? 'login_only') === 'login_only' ? 'selected' : ''; ?>>Login Only</option>
                        <option value="login_and_guest" <?php echo ($config['access_mode'] ?? '') === 'login_and_guest' ? 'selected' : ''; ?>>Login + Guest Access</option>
                    </select>
                </div>

                <div style="margin-bottom: 20px;">
                    <label>
                        <input type="checkbox" name="allow_private_messages" <?php echo ($config['allow_private_messages'] ?? true) ? 'checked' : ''; ?>>
                        <strong>Allow Private Messaging</strong>
                    </label>
                </div>

                <button type="submit" style="padding: 10px 20px;">Save Settings</button>
            </form>

            <hr>

            <!-- Online Users -->
            <h2>Online Users (<?php echo $online_count; ?>)</h2>
            <?php if (empty($online_users)): ?>
                <p>No users currently online.</p>
            <?php else: ?>
                <ul>
                    <?php foreach ($online_users as $user): ?>
                        <li>
                            <strong><?php echo e($user['display_name']); ?></strong>
                            (<?php echo e($user['role']); ?>)
                            <?php if ($user['user_type'] === 'guest'): ?>
                                <span style="color: #666;">- Guest</span>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>

            <hr>

            <!-- Recent Messages -->
            <h2>Recent Messages</h2>
            <?php if (empty($messages)): ?>
                <p>No messages yet.</p>
            <?php else: ?>
                <div style="max-height: 400px; overflow-y: auto;">
                    <?php
                    // Show last 20 messages
                    $recent_messages = array_slice(array_reverse($messages), 0, 20);
                    foreach ($recent_messages as $msg):
                    ?>
                        <div class="message-item">
                            <div class="message-header">
                                <?php echo e($msg['display_name']); ?>
                                <?php if ($msg['role'] !== 'guest'): ?>
                                    <span style="background: #0066cc; color: white; padding: 2px 6px; font-size: 0.75em;"><?php echo strtoupper(e($msg['role'])); ?></span>
                                <?php endif; ?>
                            </div>
                            <div style="margin: 5px 0;"><?php echo nl2br(e($msg['content'])); ?></div>
                            <div class="message-meta">
                                <?php echo date('M j, Y g:i A', $msg['timestamp']); ?>
                                <?php if ($is_admin): ?>
                                    <form method="POST" class="inline-form" style="margin-left: 10px;">
                                        <input type="hidden" name="action" value="delete_message">
                                        <input type="hidden" name="message_id" value="<?php echo e($msg['id']); ?>">
                                        <button type="submit" class="delete-button-small" onclick="return confirm('Delete this message?');">Delete</button>
                                    </form>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <?php if ($is_admin && !empty($messages)): ?>
                <form method="POST" style="margin-top: 15px;">
                    <input type="hidden" name="action" value="clear_all_messages">
                    <button type="submit" class="delete-button" onclick="return confirm('Delete ALL messages? This cannot be undone!');">Clear All Messages</button>
                </form>
            <?php endif; ?>

        </div>
    </div>

    <!-- Modal detection script -->
    <script>
    (function() {
        const urlParams = new URLSearchParams(window.location.search);
        const isModal = urlParams.get('modal') === '1';

        if (isModal) {
            // Hide return link if any
            const links = document.querySelectorAll('a[href="/"]');
            links.forEach(link => {
                if (link.textContent.includes('Return')) {
                    link.style.display = 'none';
                }
            });

            <?php if ($success_message): ?>
            // Auto-close modal on success
            setTimeout(() => {
                window.parent?.postMessage('plugin-admin-saved', '*');
            }, 1500);
            <?php endif; ?>

            // Modal styling
            document.body.style.background = '#008080';
            document.body.style.margin = '0';
            document.body.style.padding = '10px';
        }
    })();
    </script>
</body>
</html>
