<?php
/**
 * PlayList Embed Handler
 * Renders playlist player inline in article
 */

// Get instance context
$instance_path = $_ENV['BOOKGRAM_INSTANCE_PATH'] ?? null;
$instance_id = $_ENV['BOOKGRAM_INSTANCE_ID'] ?? null;
$page_uuid = $_ENV['BOOKGRAM_PAGE_UUID'] ?? null;

if (!$instance_path || !$instance_id) {
    echo '<p style="color: #cc0000;">Error: PlayList instance context not available.</p>';
    return;
}

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';

// Check if configured
if (!file_exists($config_file)) {
    echo '<div style="padding: 20px; background: #fff3cd; border: 2px solid #ff6600; margin: 20px 0;">';
    echo '<h3>🎵 Playlist Not Configured</h3>';
    echo '<p><a href="#" onclick="event.preventDefault(); openPluginAdmin(\'' . htmlspecialchars($page_uuid) . '\', \'plugin_PlayList\', \'' . htmlspecialchars($instance_id) . '\', \'PlayList\');" class="config-link">Configure Playlist</a></p>';
    echo '</div>';
    return;
}

// Load config
$config = json_decode(file_get_contents($config_file), true);

// Check if source is set
if ($config['source_type'] === 'none' || empty($config['source_value'])) {
    echo '<div style="padding: 20px; background: #fff3cd; border: 2px solid #ff6600; margin: 20px 0;">';
    echo '<h3>🎵 ' . htmlspecialchars($config['player_name']) . '</h3>';
    echo '<p>No playlist source configured. <a href="#" onclick="event.preventDefault(); openPluginAdmin(\'' . htmlspecialchars($page_uuid) . '\', \'plugin_PlayList\', \'' . htmlspecialchars($instance_id) . '\', \'PlayList\');" class="config-link">Add a playlist source</a></p>';
    echo '</div>';
    return;
}

// Generate unique player ID
$player_id = 'playlist_' . htmlspecialchars($instance_id);
?>

<div class="playlist-container" id="<?php echo $player_id; ?>">
    <div class="playlist-header">
        <h3>🎵 <?php echo htmlspecialchars($config['player_name']); ?></h3>
        <?php if (!empty($config['description'])): ?>
            <p><?php echo htmlspecialchars($config['description']); ?></p>
        <?php endif; ?>
    </div>

    <!-- Player Element -->
    <div class="playlist-player">
        <?php if ($config['source_type'] === 'youtube'): ?>
            <!-- YouTube embed will be handled by JavaScript -->
            <div id="<?php echo $player_id; ?>_youtube" class="youtube-player"></div>
        <?php elseif ($config['source_type'] === 'odysee'): ?>
            <!-- Odysee embed -->
            <iframe
                id="<?php echo $player_id; ?>_odysee"
                src="<?php echo htmlspecialchars($config['source_value']); ?>"
                allowfullscreen
                class="odysee-player">
            </iframe>
        <?php else: ?>
            <!-- Standard HTML5 audio/video player -->
            <audio id="<?php echo $player_id; ?>_player" controls class="media-player">
                Your browser does not support the audio element.
            </audio>
        <?php endif; ?>
    </div>

    <?php if ($config['source_type'] !== 'youtube' && $config['source_type'] !== 'odysee'): ?>
    <!-- Controls for standard playlists -->
    <div class="playlist-controls">
        <button id="<?php echo $player_id; ?>_prev" class="control-btn">⏮ Previous</button>
        <button id="<?php echo $player_id; ?>_playpause" class="control-btn">▶ Play</button>
        <button id="<?php echo $player_id; ?>_next" class="control-btn">Next ⏭</button>
        <?php if ($config['allow_speed_control']): ?>
            <button onclick="<?php echo $player_id; ?>_changeSpeed(-0.25)" class="control-btn">Speed -</button>
            <button onclick="<?php echo $player_id; ?>_changeSpeed(0.25)" class="control-btn">Speed +</button>
            <span class="speed-display">Speed: <span id="<?php echo $player_id; ?>_speed">1.00</span>x</span>
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <?php if ($config['show_metadata']): ?>
    <!-- Metadata Display -->
    <div id="<?php echo $player_id; ?>_metadata" class="playlist-metadata">
        Metadata will appear here.
    </div>
    <?php endif; ?>

    <!-- Track List -->
    <div id="<?php echo $player_id; ?>_tracklist" class="playlist-tracklist">
        Loading playlist...
    </div>

    <!-- Error Display -->
    <div id="<?php echo $player_id; ?>_error" class="playlist-error"></div>
</div>

<!-- Include external libraries -->
<script src="https://cdn.jsdelivr.net/npm/hls.js@latest"></script>
<?php if ($config['source_type'] === 'youtube'): ?>
<script src="https://www.youtube.com/iframe_api"></script>
<?php endif; ?>

<script>
// Player configuration
const <?php echo $player_id; ?>_config = {
    instanceId: '<?php echo htmlspecialchars($instance_id); ?>',
    playerId: '<?php echo $player_id; ?>',
    sourceType: '<?php echo htmlspecialchars($config['source_type']); ?>',
    sourceValue: <?php echo json_encode($config['source_value']); ?>,
    autoPlay: <?php echo $config['auto_play'] ? 'true' : 'false'; ?>,
    showMetadata: <?php echo $config['show_metadata'] ? 'true' : 'false'; ?>,
    allowSpeedControl: <?php echo $config['allow_speed_control'] ? 'true' : 'false'; ?>,
    corsProxy: '<?php echo htmlspecialchars($config['cors_proxy']); ?>'
};

// Initialize player
(function() {
    const cfg = <?php echo $player_id; ?>_config;
    let tracks = [];
    let currentIndex = 0;
    let hls = null;
    let ytPlayer = null;

    const audioElement = document.getElementById(cfg.playerId + '_player');
    const playPauseBtn = document.getElementById(cfg.playerId + '_playpause');
    const metadataDiv = document.getElementById(cfg.playerId + '_metadata');
    const errorDiv = document.getElementById(cfg.playerId + '_error');
    const tracklistDiv = document.getElementById(cfg.playerId + '_tracklist');
    const speedSpan = document.getElementById(cfg.playerId + '_speed');

    // Show error
    function showError(message) {
        if (errorDiv) {
            errorDiv.textContent = 'Error: ' + message;
            errorDiv.style.display = 'block';
        }
        console.error(message);
    }

    // Parse PLS format
    function parsePLS(text) {
        const lines = text.split('\n');
        const trackMap = {};
        let numEntries = 0;

        lines.forEach(line => {
            line = line.trim();
            if (!line || line.startsWith('[')) return;
            const [key, value] = line.split('=').map(part => part.trim());
            if (!key || !value) return;

            if (key === 'NumberOfEntries') {
                numEntries = parseInt(value, 10);
                return;
            }

            const match = key.match(/^(File|Title|Length)(\d+)$/);
            if (match) {
                const [, type, numStr] = match;
                const num = parseInt(numStr, 10);
                if (!trackMap[num]) trackMap[num] = {};
                if (type === 'File') trackMap[num].url = value;
                else if (type === 'Title') trackMap[num].title = value;
                else if (type === 'Length') trackMap[num].length = parseInt(value, 10);
            }
        });

        const trackList = [];
        for (let i = 1; i <= numEntries; i++) {
            if (trackMap[i] && trackMap[i].url) {
                trackList.push({
                    url: trackMap[i].url,
                    title: trackMap[i].title || 'Track ' + i,
                    length: trackMap[i].length || -1
                });
            }
        }
        return trackList;
    }

    // Parse M3U format
    function parseM3U(text) {
        const lines = text.split('\n');
        const trackList = [];
        let currentTrack = null;

        lines.forEach(line => {
            line = line.trim();
            if (!line || line.startsWith('#EXTM3U')) return;

            if (line.startsWith('#EXTINF:')) {
                const parts = line.split(',', 2);
                const title = parts[1] || 'Unknown';
                currentTrack = { title: title.trim(), url: null, length: -1 };
            } else if (line && !line.startsWith('#')) {
                if (currentTrack) {
                    currentTrack.url = line;
                    trackList.push(currentTrack);
                    currentTrack = null;
                } else {
                    // Direct URL without EXTINF
                    trackList.push({ url: line, title: 'Track ' + (trackList.length + 1), length: -1 });
                }
            }
        });

        return trackList;
    }

    // Load playlist
    async function loadPlaylist() {
        try {
            if (cfg.sourceType === 'youtube') {
                loadYouTube();
                return;
            } else if (cfg.sourceType === 'odysee') {
                tracklistDiv.innerHTML = '<p>Odysee video loaded.</p>';
                return;
            }

            const url = cfg.sourceValue;
            let text;

            if (cfg.sourceType === 'file') {
                // File is stored locally
                const response = await fetch(url);
                if (!response.ok) throw new Error('Failed to load playlist file');
                text = await response.text();
            } else {
                // External URL - use CORS proxy if needed
                try {
                    const response = await fetch(url);
                    text = await response.text();
                } catch (e) {
                    // Try with CORS proxy
                    const response = await fetch(cfg.corsProxy + url);
                    if (!response.ok) throw new Error('Failed to fetch playlist');
                    text = await response.text();
                }
            }

            // Parse based on extension
            if (url.toLowerCase().includes('.pls')) {
                tracks = parsePLS(text);
            } else if (url.toLowerCase().includes('.m3u')) {
                tracks = parseM3U(text);
            } else {
                // Single file
                tracks = [{ url: url, title: 'Media File', length: -1 }];
            }

            if (tracks.length === 0) {
                throw new Error('No tracks found in playlist');
            }

            renderTrackList();
            if (cfg.autoPlay) {
                playTrack(0);
            } else {
                updateMetadata(0);
            }
        } catch (error) {
            showError(error.message);
        }
    }

    // Render track list
    function renderTrackList() {
        if (!tracklistDiv) return;

        let html = '<div class="track-list"><h4>Tracks:</h4><ul>';
        tracks.forEach((track, idx) => {
            html += '<li><a href="#" onclick="event.preventDefault(); playTrack_' + cfg.instanceId + '(' + idx + ');">' +
                    (idx + 1) + '. ' + escapeHtml(track.title) + '</a></li>';
        });
        html += '</ul></div>';
        tracklistDiv.innerHTML = html;
    }

    // Escape HTML
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Play track
    window['playTrack_' + cfg.instanceId] = function(index) {
        if (index < 0 || index >= tracks.length) return;
        currentIndex = index;
        const track = tracks[index];

        if (errorDiv) errorDiv.style.display = 'none';

        if (track.url.endsWith('.m3u8') && typeof Hls !== 'undefined' && Hls.isSupported()) {
            if (hls) hls.destroy();
            hls = new Hls();
            hls.loadSource(track.url);
            hls.attachMedia(audioElement);
            hls.on(Hls.Events.ERROR, (event, data) => {
                if (data.fatal) {
                    showError('HLS Error: ' + data.type);
                }
            });
            audioElement.play().catch(error => showError('Playback error: ' + error.message));
        } else {
            audioElement.src = track.url;
            audioElement.play().catch(error => showError('Playback error: ' + error.message));
        }

        if (playPauseBtn) playPauseBtn.textContent = '⏸ Pause';
        updateMetadata(index);
    };

    // Update metadata
    function updateMetadata(index) {
        if (!cfg.showMetadata || !metadataDiv) return;

        const track = tracks[index];
        let html = '<strong>Track ' + (index + 1) + ' of ' + tracks.length + ':</strong> ' + escapeHtml(track.title);
        if (track.length > 0) {
            html += '<br><strong>Length:</strong> ' + track.length + ' seconds';
        }
        metadataDiv.innerHTML = html;
    }

    // YouTube support
    function loadYouTube() {
        const ytInfo = extractYouTubeInfo(cfg.sourceValue);
        if (!ytInfo.videoId && !ytInfo.playlistId) {
            showError('Invalid YouTube URL');
            return;
        }

        // Wait for YouTube API to load
        if (typeof YT === 'undefined' || !YT.Player) {
            setTimeout(loadYouTube, 100);
            return;
        }

        const playerConfig = {
            height: '315',
            width: '560',
            playerVars: {
                autoplay: cfg.autoPlay ? 1 : 0
            },
            events: {
                'onReady': onYouTubeReady,
                'onStateChange': onYouTubeStateChange
            }
        };

        // Handle playlist or single video
        if (ytInfo.playlistId) {
            playerConfig.playerVars.list = ytInfo.playlistId;
            if (ytInfo.videoId) {
                // Playlist with starting video
                playerConfig.videoId = ytInfo.videoId;
            }
            if (tracklistDiv) {
                tracklistDiv.innerHTML = '<p>YouTube playlist loaded. Metadata will update as videos play.</p>';
            }
        } else {
            // Single video
            playerConfig.videoId = ytInfo.videoId;
            if (tracklistDiv) {
                tracklistDiv.innerHTML = '<p>YouTube video loaded.</p>';
            }
        }

        ytPlayer = new YT.Player(cfg.playerId + '_youtube', playerConfig);
    }

    // YouTube player ready event
    function onYouTubeReady(event) {
        updateYouTubeMetadata();

        // For playlists, show playlist info
        if (ytPlayer && typeof ytPlayer.getPlaylist === 'function') {
            const playlist = ytPlayer.getPlaylist();
            if (playlist && playlist.length > 0 && tracklistDiv) {
                tracklistDiv.innerHTML = '<p><strong>Playlist:</strong> ' + playlist.length + ' videos</p>';
            }
        }
    }

    // YouTube state change event (fires when video changes, plays, pauses, etc.)
    function onYouTubeStateChange(event) {
        // Update metadata when video starts playing or is cued
        if (event.data === YT.PlayerState.PLAYING || event.data === YT.PlayerState.CUED) {
            updateYouTubeMetadata();
        }
    }

    // Update YouTube metadata display
    function updateYouTubeMetadata() {
        if (!cfg.showMetadata || !metadataDiv || !ytPlayer) return;

        try {
            const videoData = ytPlayer.getVideoData();
            if (videoData && videoData.title) {
                let html = '<strong>Now Playing:</strong> ' + escapeHtml(videoData.title);

                if (videoData.author) {
                    html += '<br><strong>Channel:</strong> ' + escapeHtml(videoData.author);
                }

                // Show playlist position if available
                if (typeof ytPlayer.getPlaylistIndex === 'function') {
                    const playlistIndex = ytPlayer.getPlaylistIndex();
                    if (playlistIndex >= 0) {
                        const playlist = ytPlayer.getPlaylist();
                        if (playlist && playlist.length > 0) {
                            html += '<br><strong>Position:</strong> Video ' + (playlistIndex + 1) + ' of ' + playlist.length;
                        }
                    }
                }

                // Show video duration if available
                if (typeof ytPlayer.getDuration === 'function') {
                    const duration = ytPlayer.getDuration();
                    if (duration > 0) {
                        const minutes = Math.floor(duration / 60);
                        const seconds = Math.floor(duration % 60);
                        html += '<br><strong>Duration:</strong> ' + minutes + ':' + (seconds < 10 ? '0' : '') + seconds;
                    }
                }

                metadataDiv.innerHTML = html;
            } else {
                metadataDiv.innerHTML = 'Loading metadata...';
            }
        } catch (error) {
            console.error('Error updating YouTube metadata:', error);
            if (metadataDiv) {
                metadataDiv.innerHTML = 'YouTube video playing';
            }
        }
    }

    // Extract YouTube video ID and/or playlist ID
    function extractYouTubeInfo(url) {
        const result = { videoId: null, playlistId: null };

        // Extract video ID from watch URL or short URL
        const videoMatch = url.match(/(?:youtube\.com\/watch\?v=|youtu\.be\/)([a-zA-Z0-9_-]+)/);
        if (videoMatch) {
            result.videoId = videoMatch[1];
        }

        // Extract playlist ID
        const playlistMatch = url.match(/[?&]list=([a-zA-Z0-9_-]+)/);
        if (playlistMatch) {
            result.playlistId = playlistMatch[1];
        }

        // Check for direct playlist URL
        if (!result.videoId && !result.playlistId) {
            const directPlaylistMatch = url.match(/youtube\.com\/playlist\?list=([a-zA-Z0-9_-]+)/);
            if (directPlaylistMatch) {
                result.playlistId = directPlaylistMatch[1];
            }
        }

        return result;
    }

    // Controls
    if (playPauseBtn) {
        playPauseBtn.addEventListener('click', () => {
            if (audioElement.paused) {
                audioElement.play();
                playPauseBtn.textContent = '⏸ Pause';
            } else {
                audioElement.pause();
                playPauseBtn.textContent = '▶ Play';
            }
        });
    }

    const prevBtn = document.getElementById(cfg.playerId + '_prev');
    if (prevBtn) {
        prevBtn.addEventListener('click', () => window['playTrack_' + cfg.instanceId](currentIndex - 1));
    }

    const nextBtn = document.getElementById(cfg.playerId + '_next');
    if (nextBtn) {
        nextBtn.addEventListener('click', () => window['playTrack_' + cfg.instanceId](currentIndex + 1));
    }

    if (audioElement) {
        audioElement.addEventListener('ended', () => {
            window['playTrack_' + cfg.instanceId](currentIndex + 1);
        });
    }

    // Speed control
    window[cfg.playerId + '_changeSpeed'] = function(delta) {
        if (!cfg.allowSpeedControl || !audioElement) return;
        audioElement.playbackRate = Math.max(0.25, Math.min(4, audioElement.playbackRate + delta));
        if (speedSpan) speedSpan.textContent = audioElement.playbackRate.toFixed(2);
    };

    // Load on init
    loadPlaylist();
})();
</script>
