<?php
/**
 * PlayList Admin Panel
 * Configure playlist sources
 */

// Use BookGram authentication
require_once dirname(dirname(__DIR__)) . '/app/core/plugin_auth.php';
$auth = bookgram_auth_check();
bookgram_require_editor();

// Get instance context
$instance_id = bookgram_get_instance_id();
$instance_path = bookgram_get_instance_path();

// Validate instance path
if (!$instance_path || !is_dir($instance_path)) {
    die('<html><body><h1>Error</h1><p>Instance path not found.</p></body></html>');
}

$data_dir = $instance_path . '/data/';
$config_file = $data_dir . 'config.json';
$uploads_dir = $data_dir . 'uploads/';

// Load config
$config = file_exists($config_file) ? json_decode(file_get_contents($config_file), true) : [];

// Handle settings update
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'update_config') {
        $config['player_name'] = trim($_POST['player_name'] ?? 'Playlist Player');
        $config['description'] = trim($_POST['description'] ?? '');
        $config['source_type'] = $_POST['source_type'] ?? 'none';
        $config['auto_play'] = isset($_POST['auto_play']);
        $config['show_metadata'] = isset($_POST['show_metadata']);
        $config['allow_speed_control'] = isset($_POST['allow_speed_control']);
        $config['cors_proxy'] = trim($_POST['cors_proxy'] ?? 'https://cors-anywhere.herokuapp.com/');

        // Handle different source types
        if ($config['source_type'] === 'url') {
            $config['source_value'] = trim($_POST['source_url'] ?? '');
        } elseif ($config['source_type'] === 'youtube') {
            $config['source_value'] = trim($_POST['source_youtube'] ?? '');
        } elseif ($config['source_type'] === 'odysee') {
            $config['source_value'] = trim($_POST['source_odysee'] ?? '');
        } elseif ($config['source_type'] === 'file') {
            // Handle file upload
            if (!empty($_FILES['playlist_file']['tmp_name'])) {
                $upload_file = $_FILES['playlist_file'];
                $filename = basename($upload_file['name']);
                $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

                if (!in_array($ext, ['m3u', 'm3u8', 'pls'])) {
                    $error_message = 'Invalid file type. Only M3U, M3U8, and PLS files are allowed.';
                } elseif ($upload_file['size'] > 5000000) { // 5MB limit
                    $error_message = 'File too large. Maximum 5MB.';
                } else {
                    $dest_path = $uploads_dir . 'playlist.' . $ext;
                    if (move_uploaded_file($upload_file['tmp_name'], $dest_path)) {
                        // Generate relative URL
                        $config['source_value'] = '/app/storage/plugin_instances/' .
                                                 basename(dirname(dirname($instance_path))) .
                                                 '/plugin_PlayList/' . $instance_id .
                                                 '/data/uploads/playlist.' . $ext;
                        $success_message = 'Playlist file uploaded successfully!';
                    } else {
                        $error_message = 'Failed to upload file.';
                    }
                }
            } elseif (empty($config['source_value'])) {
                $error_message = 'Please upload a playlist file.';
            }
        } else {
            $config['source_value'] = '';
        }

        if (empty($error_message)) {
            if (file_put_contents($config_file, json_encode($config, JSON_PRETTY_PRINT))) {
                if (empty($success_message)) {
                    $success_message = 'Configuration updated successfully!';
                }
            } else {
                $error_message = 'Failed to save configuration.';
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PlayList Admin - <?php echo htmlspecialchars($instance_id); ?></title>
    <link rel="stylesheet" href="/public/css/os2-theme.css">
    <style>
        body { background: #008080; padding: 20px; }
        .admin-container { max-width: 900px; margin: 0 auto; }
        .form-group { margin-bottom: 20px; }
        .form-group label { display: block; font-weight: bold; margin-bottom: 5px; color: #000080; }
        .form-group input[type="text"],
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 8px;
            border: 2px inset #c0c0c0;
            font-family: 'Courier New', monospace;
            background: #ffffff;
        }
        .form-group textarea { min-height: 60px; }
        .source-section {
            display: none;
            padding: 15px;
            background: #e0e0e0;
            border: 2px inset #c0c0c0;
            margin-top: 10px;
        }
        .source-section.active { display: block; }
        .info-box { background: #e7f3ff; padding: 15px; border-left: 4px solid #007bff; margin-bottom: 20px; }
        .help-text { font-size: 0.9em; color: #666; margin-top: 5px; }
        .checkbox-group { margin: 10px 0; }
    </style>
</head>
<body>
    <div class="admin-container">
        <div class="window">
            <div class="window-title-bar">
                <span class="title">PlayList Admin - <?php echo htmlspecialchars($config['player_name'] ?? 'Playlist Player'); ?></span>
            </div>
            <div class="window-content">
                <div class="info-box">
                    <strong>Instance:</strong> <?php echo htmlspecialchars($instance_id); ?><br>
                    <strong>Manager:</strong> <?php echo htmlspecialchars(bookgram_get_username()); ?> (<?php echo bookgram_is_admin() ? 'Admin' : 'Editor'; ?>)
                </div>

                <?php if ($success_message): ?>
                    <div style="background: #ccffcc; padding: 10px; border: 2px solid #00cc00; margin-bottom: 20px;">
                        ✓ <?php echo htmlspecialchars($success_message); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error_message): ?>
                    <div style="background: #ffcccc; padding: 10px; border: 2px solid #cc0000; margin-bottom: 20px;">
                        ✗ <?php echo htmlspecialchars($error_message); ?>
                    </div>
                <?php endif; ?>

                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="update_config">

                    <h2>Basic Settings</h2>

                    <div class="form-group">
                        <label>Player Name:</label>
                        <input type="text" name="player_name" value="<?php echo htmlspecialchars($config['player_name'] ?? 'Playlist Player'); ?>" required>
                    </div>

                    <div class="form-group">
                        <label>Description (optional):</label>
                        <textarea name="description"><?php echo htmlspecialchars($config['description'] ?? ''); ?></textarea>
                    </div>

                    <h2>Playlist Source</h2>

                    <div class="form-group">
                        <label>Source Type:</label>
                        <select name="source_type" id="source_type" onchange="toggleSourceSections()">
                            <option value="none" <?php echo ($config['source_type'] ?? 'none') === 'none' ? 'selected' : ''; ?>>None (Not configured)</option>
                            <option value="url" <?php echo ($config['source_type'] ?? '') === 'url' ? 'selected' : ''; ?>>Direct URL (M3U/PLS/Media file)</option>
                            <option value="file" <?php echo ($config['source_type'] ?? '') === 'file' ? 'selected' : ''; ?>>Upload Playlist File</option>
                            <option value="youtube" <?php echo ($config['source_type'] ?? '') === 'youtube' ? 'selected' : ''; ?>>YouTube Video/Playlist</option>
                            <option value="odysee" <?php echo ($config['source_type'] ?? '') === 'odysee' ? 'selected' : ''; ?>>Odysee Video</option>
                        </select>
                    </div>

                    <!-- URL Source -->
                    <div class="source-section" id="source_url_section">
                        <label>Playlist/Media URL:</label>
                        <input type="text" name="source_url" value="<?php echo htmlspecialchars($config['source_type'] === 'url' ? ($config['source_value'] ?? '') : ''); ?>" placeholder="https://example.com/playlist.m3u">
                        <div class="help-text">
                            Direct URL to M3U, M3U8, PLS playlist file, or direct audio/video file.<br>
                            Examples: https://example.com/stream.m3u, https://example.com/audio.mp3
                        </div>
                    </div>

                    <!-- File Upload Source -->
                    <div class="source-section" id="source_file_section">
                        <label>Upload Playlist File:</label>
                        <input type="file" name="playlist_file" accept=".m3u,.m3u8,.pls">
                        <div class="help-text">
                            Upload an M3U, M3U8, or PLS playlist file (max 5MB).<br>
                            <?php if ($config['source_type'] === 'file' && !empty($config['source_value'])): ?>
                                <strong>Current file:</strong> <?php echo basename($config['source_value']); ?>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- YouTube Source -->
                    <div class="source-section" id="source_youtube_section">
                        <label>YouTube URL:</label>
                        <input type="text" name="source_youtube" value="<?php echo htmlspecialchars($config['source_type'] === 'youtube' ? ($config['source_value'] ?? '') : ''); ?>" placeholder="https://www.youtube.com/watch?v=VIDEO_ID or playlist URL">
                        <div class="help-text">
                            YouTube video or playlist URL.<br>
                            <strong>Video:</strong> https://www.youtube.com/watch?v=dQw4w9WgXcQ or https://youtu.be/dQw4w9WgXcQ<br>
                            <strong>Playlist:</strong> https://www.youtube.com/playlist?list=PLAYLIST_ID<br>
                            <strong>Video in playlist:</strong> https://www.youtube.com/watch?v=VIDEO_ID&list=PLAYLIST_ID
                        </div>
                    </div>

                    <!-- Odysee Source -->
                    <div class="source-section" id="source_odysee_section">
                        <label>Odysee Embed URL:</label>
                        <input type="text" name="source_odysee" value="<?php echo htmlspecialchars($config['source_type'] === 'odysee' ? ($config['source_value'] ?? '') : ''); ?>" placeholder="https://odysee.com/$/embed/@channel/video">
                        <div class="help-text">
                            Odysee embed URL. Get this by clicking "Share" → "Embed" on Odysee.<br>
                            Example: https://odysee.com/$/embed/@channel:a/video-name:b
                        </div>
                    </div>

                    <h2>Player Options</h2>

                    <div class="checkbox-group">
                        <label>
                            <input type="checkbox" name="auto_play" value="1" <?php echo !empty($config['auto_play']) ? 'checked' : ''; ?>>
                            Auto-play on load
                        </label>
                    </div>

                    <div class="checkbox-group">
                        <label>
                            <input type="checkbox" name="show_metadata" value="1" <?php echo ($config['show_metadata'] ?? true) ? 'checked' : ''; ?>>
                            Show metadata/track info
                        </label>
                    </div>

                    <div class="checkbox-group">
                        <label>
                            <input type="checkbox" name="allow_speed_control" value="1" <?php echo ($config['allow_speed_control'] ?? true) ? 'checked' : ''; ?>>
                            Allow playback speed control
                        </label>
                    </div>

                    <h2>Advanced Settings</h2>

                    <div class="form-group">
                        <label>CORS Proxy (for external playlists):</label>
                        <input type="text" name="cors_proxy" value="<?php echo htmlspecialchars($config['cors_proxy'] ?? 'https://cors-anywhere.herokuapp.com/'); ?>">
                        <div class="help-text">
                            CORS proxy URL for loading external playlists. Leave default unless you have your own proxy.
                        </div>
                    </div>

                    <button type="submit" style="padding: 10px 30px; background: #000080; color: #fff; border: 2px outset #c0c0c0; cursor: pointer; font-size: 16px; margin-top: 20px;">
                        Save Configuration
                    </button>
                </form>

                <p style="margin-top: 30px;">
                    <a href="/">← Return to BookGram</a>
                </p>
            </div>
        </div>
    </div>

    <script>
    function toggleSourceSections() {
        const sourceType = document.getElementById('source_type').value;
        const sections = document.querySelectorAll('.source-section');

        sections.forEach(section => {
            section.classList.remove('active');
        });

        if (sourceType !== 'none') {
            const activeSection = document.getElementById('source_' + sourceType + '_section');
            if (activeSection) {
                activeSection.classList.add('active');
            }
        }
    }

    // Initialize on page load
    toggleSourceSections();

    // Modal mode detection and parent notification
    (function() {
        const urlParams = new URLSearchParams(window.location.search);
        const isModal = urlParams.get('modal') === '1';

        if (isModal) {
            // Hide "Return to BookGram" link in modal mode
            const returnLink = document.querySelector('a[href="/"]');
            if (returnLink && returnLink.textContent.includes('Return to BookGram')) {
                returnLink.style.display = 'none';
            }

            // Notify parent on successful save
            <?php if ($success_message): ?>
            setTimeout(function() {
                if (window.parent && window.parent !== window) {
                    window.parent.postMessage('plugin-admin-saved', '*');
                }
            }, 1500); // Wait 1.5 seconds to show success message
            <?php endif; ?>

            // Add modal styling adjustments
            document.body.style.margin = '0';
            document.body.style.padding = '10px';
            document.body.style.background = '#008080';
        }
    })();
    </script>
</body>
</html>
