# YouTube Playlist Support Fix

## Problem
The PlayList plugin was showing "Invalid YouTube URL" error when trying to use a YouTube playlist URL like:
```
https://www.youtube.com/playlist?list=PLw3nL3S4fLQPyjMWHWg5uzOHtr7Ar9JNT
```

## Root Cause
The `extractYouTubeId()` function in `embed.php` only supported individual video URLs:
- `https://www.youtube.com/watch?v=VIDEO_ID`
- `https://youtu.be/VIDEO_ID`

It did not recognize:
- Playlist URLs: `https://www.youtube.com/playlist?list=PLAYLIST_ID`
- Mixed URLs: `https://www.youtube.com/watch?v=VIDEO_ID&list=PLAYLIST_ID`

## Solution

### 1. Enhanced URL Parsing
Replaced `extractYouTubeId()` with `extractYouTubeInfo()` that returns both video and playlist IDs:

```javascript
function extractYouTubeInfo(url) {
    const result = { videoId: null, playlistId: null };

    // Extract video ID from watch URL or short URL
    const videoMatch = url.match(/(?:youtube\.com\/watch\?v=|youtu\.be\/)([a-zA-Z0-9_-]+)/);
    if (videoMatch) {
        result.videoId = videoMatch[1];
    }

    // Extract playlist ID
    const playlistMatch = url.match(/[?&]list=([a-zA-Z0-9_-]+)/);
    if (playlistMatch) {
        result.playlistId = playlistMatch[1];
    }

    // Check for direct playlist URL
    if (!result.videoId && !result.playlistId) {
        const directPlaylistMatch = url.match(/youtube\.com\/playlist\?list=([a-zA-Z0-9_-]+)/);
        if (directPlaylistMatch) {
            result.playlistId = directPlaylistMatch[1];
        }
    }

    return result;
}
```

### 2. Updated YouTube Player Initialization
Modified `loadYouTube()` to handle playlists using the YouTube IFrame API's `list` parameter:

```javascript
function loadYouTube() {
    const ytInfo = extractYouTubeInfo(cfg.sourceValue);
    if (!ytInfo.videoId && !ytInfo.playlistId) {
        showError('Invalid YouTube URL');
        return;
    }

    const playerConfig = {
        height: '315',
        width: '560',
        playerVars: {
            autoplay: cfg.autoPlay ? 1 : 0
        }
    };

    // Handle playlist or single video
    if (ytInfo.playlistId) {
        playerConfig.playerVars.list = ytInfo.playlistId;
        if (ytInfo.videoId) {
            // Playlist with starting video
            playerConfig.videoId = ytInfo.videoId;
        }
        tracklistDiv.innerHTML = '<p>YouTube playlist loaded.</p>';
    } else {
        // Single video
        playerConfig.videoId = ytInfo.videoId;
        tracklistDiv.innerHTML = '<p>YouTube video loaded.</p>';
    }

    ytPlayer = new YT.Player(cfg.playerId + '_youtube', playerConfig);
}
```

## Supported URL Formats

### ✅ Now Works
1. **Single video:**
   - `https://www.youtube.com/watch?v=dQw4w9WgXcQ`
   - `https://youtu.be/dQw4w9WgXcQ`

2. **Playlist only:**
   - `https://www.youtube.com/playlist?list=PLw3nL3S4fLQPyjMWHWg5uzOHtr7Ar9JNT` ⭐ **FIXED**

3. **Video within playlist:**
   - `https://www.youtube.com/watch?v=dQw4w9WgXcQ&list=PLw3nL3S4fLQPyjMWHWg5uzOHtr7Ar9JNT`
   - Starts at the specific video, then continues with playlist

## Testing

A test file has been created at `test_youtube_urls.html` to verify the URL parsing logic works correctly. Open it in a browser to see:

- Video ID extraction
- Playlist ID extraction
- Mixed URL handling
- Your specific URL validation

## Files Modified

1. **embed.php** (lines 334-398)
   - Replaced `extractYouTubeId()` with `extractYouTubeInfo()`
   - Updated `loadYouTube()` to handle playlists

2. **admin/index.php** (lines 202-211)
   - Updated help text to show playlist support
   - Added examples for all URL formats

3. **README.md**
   - Updated YouTube section with playlist examples
   - Moved playlist support from "Future Enhancements" to current features
   - Updated version history to 2.0.1

4. **plugin.json**
   - Bumped version to 2.0.1

## Usage

In the PlayList admin panel:

1. Select **Source Type:** YouTube Video/Playlist
2. Enter any of these URL formats:
   - Playlist: `https://www.youtube.com/playlist?list=YOUR_PLAYLIST_ID`
   - Video: `https://www.youtube.com/watch?v=YOUR_VIDEO_ID`
   - Video in playlist: `https://www.youtube.com/watch?v=VIDEO_ID&list=PLAYLIST_ID`
3. Save configuration
4. The playlist will now load correctly in the article

## Additional Notes

- The YouTube IFrame API automatically handles playlist navigation
- Users can navigate between videos using YouTube's built-in controls
- Playlist videos play sequentially with no additional code needed
- Auto-play setting works for both videos and playlists
