# YouTube Metadata Display Fix

## Problem
When using YouTube playlists (or videos), the metadata section was not showing any information about the currently playing video.

## Root Cause
The YouTube player was being initialized without event handlers, so the plugin couldn't:
- Detect when the player was ready
- Detect when videos changed in a playlist
- Access video information from the YouTube API

The metadata display logic only existed for M3U/PLS playlists, not for YouTube content.

## Solution

### 1. Added YouTube IFrame API Event Handlers

Updated the player configuration to include event callbacks:

```javascript
const playerConfig = {
    height: '315',
    width: '560',
    playerVars: {
        autoplay: cfg.autoPlay ? 1 : 0
    },
    events: {
        'onReady': onYouTubeReady,          // Fires when player loads
        'onStateChange': onYouTubeStateChange // Fires when video plays/pauses/changes
    }
};
```

### 2. Implemented Player Ready Handler

```javascript
function onYouTubeReady(event) {
    updateYouTubeMetadata();

    // For playlists, show playlist info
    if (ytPlayer && typeof ytPlayer.getPlaylist === 'function') {
        const playlist = ytPlayer.getPlaylist();
        if (playlist && playlist.length > 0 && tracklistDiv) {
            tracklistDiv.innerHTML = '<p><strong>Playlist:</strong> ' +
                                    playlist.length + ' videos</p>';
        }
    }
}
```

### 3. Implemented State Change Handler

```javascript
function onYouTubeStateChange(event) {
    // Update metadata when video starts playing or is cued
    if (event.data === YT.PlayerState.PLAYING ||
        event.data === YT.PlayerState.CUED) {
        updateYouTubeMetadata();
    }
}
```

### 4. Created Metadata Update Function

```javascript
function updateYouTubeMetadata() {
    if (!cfg.showMetadata || !metadataDiv || !ytPlayer) return;

    try {
        const videoData = ytPlayer.getVideoData();
        if (videoData && videoData.title) {
            let html = '<strong>Now Playing:</strong> ' + escapeHtml(videoData.title);

            if (videoData.author) {
                html += '<br><strong>Channel:</strong> ' + escapeHtml(videoData.author);
            }

            // Show playlist position if available
            if (typeof ytPlayer.getPlaylistIndex === 'function') {
                const playlistIndex = ytPlayer.getPlaylistIndex();
                if (playlistIndex >= 0) {
                    const playlist = ytPlayer.getPlaylist();
                    if (playlist && playlist.length > 0) {
                        html += '<br><strong>Position:</strong> Video ' +
                                (playlistIndex + 1) + ' of ' + playlist.length;
                    }
                }
            }

            // Show video duration
            if (typeof ytPlayer.getDuration === 'function') {
                const duration = ytPlayer.getDuration();
                if (duration > 0) {
                    const minutes = Math.floor(duration / 60);
                    const seconds = Math.floor(duration % 60);
                    html += '<br><strong>Duration:</strong> ' + minutes + ':' +
                            (seconds < 10 ? '0' : '') + seconds;
                }
            }

            metadataDiv.innerHTML = html;
        }
    } catch (error) {
        console.error('Error updating YouTube metadata:', error);
        if (metadataDiv) {
            metadataDiv.innerHTML = 'YouTube video playing';
        }
    }
}
```

## What's Now Displayed

### For Single Videos:
```
Now Playing: Video Title Here
Channel: Channel Name
Duration: 5:23
```

### For Playlists:
```
Now Playing: Current Video Title
Channel: Channel Name
Position: Video 3 of 15
Duration: 4:12
```

**Plus:** Below the player, it shows:
```
Playlist: 15 videos
```

## YouTube IFrame API Methods Used

- **`getVideoData()`** - Returns:
  - `video_id`: The YouTube video ID
  - `title`: The video title
  - `author`: The channel name

- **`getPlaylist()`** - Returns array of video IDs in the playlist

- **`getPlaylistIndex()`** - Returns current video position (0-based)

- **`getDuration()`** - Returns video length in seconds

## Behavior

### Single Video
- Metadata loads immediately when player is ready
- Shows video title, channel, and duration

### Playlist
- Metadata loads when first video is ready
- **Automatically updates** when user navigates to next/previous video
- Shows current position in playlist (e.g., "Video 5 of 20")
- Updates on every video change

## Configuration Control

Metadata display respects the "Show metadata/track info" setting in the admin panel:
- **Enabled** (default): Shows all metadata
- **Disabled**: Metadata section is hidden

## Error Handling

If the YouTube API fails to provide data:
- Falls back to "Loading metadata..." message
- Catches and logs errors to browser console
- Displays generic "YouTube video playing" if all else fails

## Browser Compatibility

Works in all modern browsers that support:
- YouTube IFrame Player API
- ES6 JavaScript features
- Promise-based API calls

## Testing

To verify metadata is working:
1. Add a YouTube playlist to your PlayList instance
2. Enable "Show metadata/track info" in admin panel
3. Load the article with the playlist
4. Check that metadata appears below the player
5. Navigate to next video in playlist
6. Verify metadata updates automatically

## Files Modified

1. **embed.php** (lines 334-446)
   - Added event handlers to player config
   - Implemented `onYouTubeReady()` callback
   - Implemented `onYouTubeStateChange()` callback
   - Created `updateYouTubeMetadata()` function

2. **plugin.json**
   - Bumped version to 2.0.2

3. **README.md**
   - Updated version history with metadata support

## Technical Notes

- The YouTube IFrame API loads asynchronously, so we wait for `YT.Player` to be available
- Metadata updates are triggered by state changes (PLAYING and CUED states)
- For playlists, each video change triggers a new metadata update
- The `escapeHtml()` function protects against XSS in video titles
- All YouTube API calls are wrapped in try-catch for safety

## Future Enhancements

Possible improvements:
- Show thumbnail images from YouTube
- Display view count and upload date
- Show all playlist video titles in a list
- Add "Copy video URL" button
- Show video description (requires additional API calls)
