<?php
/**
 * Fractal Pattern Visualization
 * Demonstrates pedagogical fractal pattern analysis with interactive parameters
 *
 * BOOKGRAM_PARAM: micro | select | HUMILITY,RESPECT,LOVE,BRAVERY,HONESTY,WISDOM,TRUTH | WISDOM
 * BOOKGRAM_PARAM: innerMicro | select | HUMILITY,RESPECT,LOVE,BRAVERY,HONESTY,WISDOM,TRUTH | HUMILITY
 * BOOKGRAM_PARAM: innerMeso | select | HUMILITY,RESPECT,LOVE,BRAVERY,HONESTY,WISDOM,TRUTH | RESPECT
 * BOOKGRAM_PARAM: meso | select | HUMILITY,RESPECT,LOVE,BRAVERY,HONESTY,WISDOM,TRUTH | WISDOM
 * BOOKGRAM_PARAM: outerMeso | select | HUMILITY,RESPECT,LOVE,BRAVERY,HONESTY,WISDOM,TRUTH | TRUTH
 * BOOKGRAM_PARAM: outerMacro | select | HUMILITY,RESPECT,LOVE,BRAVERY,HONESTY,WISDOM,TRUTH | LOVE
 * BOOKGRAM_PARAM: macro | select | HUMILITY,RESPECT,LOVE,BRAVERY,HONESTY,WISDOM,TRUTH | WISDOM
 */

// Get parameters with defaults (extracted by executor)
$micro = $micro ?? 'WISDOM';
$innerMicro = $innerMicro ?? 'HUMILITY';
$innerMeso = $innerMeso ?? 'RESPECT';
$meso = $meso ?? 'WISDOM';
$outerMeso = $outerMeso ?? 'TRUTH';
$outerMacro = $outerMacro ?? 'LOVE';
$macro = $macro ?? 'WISDOM';

// Include class definitions (adapted from original, without namespace)

abstract class FractalPattern {
    abstract public function getMicro();
    abstract public function getInnerMicro();
    abstract public function getInnerMeso();
    abstract public function getMeso();
    abstract public function getOuterMeso();
    abstract public function getOuterMacro();
    abstract public function getMacro();
    abstract public function isSelfSimilar();
    abstract public function getEmergentProperties();
}

final class Teaching {
    const HUMILITY = 'HUMILITY';
    const RESPECT = 'RESPECT';
    const LOVE = 'LOVE';
    const BRAVERY = 'BRAVERY';
    const HONESTY = 'HONESTY';
    const WISDOM = 'WISDOM';
    const TRUTH = 'TRUTH';

    private static $values = [
        self::HUMILITY,
        self::RESPECT,
        self::LOVE,
        self::BRAVERY,
        self::HONESTY,
        self::WISDOM,
        self::TRUTH
    ];

    public static function values() {
        return self::$values;
    }

    public static function isValid($value) {
        return in_array($value, self::$values);
    }

    public static function getOpposite($teaching) {
        $opposites = [
            self::WISDOM => TeachingOpposite::FOOLISHNESS,
            self::LOVE => TeachingOpposite::HATRED_APATHY,
            self::RESPECT => TeachingOpposite::ARROGANCE_DISRESPECT,
            self::BRAVERY => TeachingOpposite::COWARDICE,
            self::HUMILITY => TeachingOpposite::PRIDE_HUBRIS,
            self::HONESTY => TeachingOpposite::DECEIT,
            self::TRUTH => TeachingOpposite::UNTRUTH
        ];

        return $opposites[$teaching] ?? TeachingOpposite::UNKNOWN;
    }
}

final class TeachingOpposite {
    const FOOLISHNESS = 'FOOLISHNESS';
    const HATRED_APATHY = 'HATRED_APATHY';
    const ARROGANCE_DISRESPECT = 'ARROGANCE_DISRESPECT';
    const COWARDICE = 'COWARDICE';
    const PRIDE_HUBRIS = 'PRIDE_HUBRIS';
    const DECEIT = 'DECEIT';
    const UNTRUTH = 'UNTRUTH';
    const UNKNOWN = 'UNKNOWN';
}

class PedagogicalFractal extends FractalPattern {
    private $micro;
    private $innerMicro;
    private $innerMeso;
    private $meso;
    private $outerMeso;
    private $outerMacro;
    private $macro;

    public function __construct($micro, $innerMicro, $innerMeso, $meso, $outerMeso, $outerMacro, $macro) {
        $this->validateTeaching($micro);
        $this->validateTeaching($innerMicro);
        $this->validateTeaching($innerMeso);
        $this->validateTeaching($meso);
        $this->validateTeaching($outerMeso);
        $this->validateTeaching($outerMacro);
        $this->validateTeaching($macro);

        $this->micro = $micro;
        $this->innerMicro = $innerMicro;
        $this->innerMeso = $innerMeso;
        $this->meso = $meso;
        $this->outerMeso = $outerMeso;
        $this->outerMacro = $outerMacro;
        $this->macro = $macro;
    }

    private function validateTeaching($teaching) {
        if (!Teaching::isValid($teaching)) {
            throw new InvalidArgumentException("Invalid teaching: {$teaching}");
        }
    }

    public function getMicro() { return $this->micro; }
    public function getInnerMicro() { return $this->innerMicro; }
    public function getInnerMeso() { return $this->innerMeso; }
    public function getMeso() { return $this->meso; }
    public function getOuterMeso() { return $this->outerMeso; }
    public function getOuterMacro() { return $this->outerMacro; }
    public function getMacro() { return $this->macro; }

    public function isSelfSimilar() {
        return ($this->micro == $this->innerMicro &&
                $this->innerMicro == $this->innerMeso &&
                $this->innerMeso == $this->meso &&
                $this->meso == $this->outerMeso &&
                $this->outerMeso == $this->outerMacro &&
                $this->outerMacro == $this->macro);
    }

    public function getEmergentProperties() {
        $emergent = [];

        if ($this->isSelfSimilar() && $this->micro == Teaching::HUMILITY) {
            $emergent[] = Teaching::WISDOM;
        }

        if ($this->micro == Teaching::LOVE &&
            $this->meso == Teaching::RESPECT &&
            $this->macro == Teaching::TRUTH) {
            $emergent[] = Teaching::LOVE;
        }

        if ($this->micro == Teaching::BRAVERY &&
            $this->macro == Teaching::WISDOM) {
            $emergent[] = Teaching::TRUTH;
        }

        return array_unique($emergent);
    }

    public function analyzeFractalHealth() {
        $contradictions = $this->findContradictions();
        $coherenceScore = $this->calculateCoherenceScore();
        $healingPotential = $this->assessHealingPotential();

        return [
            'self_similar' => $this->isSelfSimilar(),
            'emergent_properties' => $this->getEmergentProperties(),
            'contradictions' => $contradictions,
            'coherence_score' => $coherenceScore,
            'healing_potential' => $healingPotential,
            'scale_integration' => $this->assessScaleIntegration(),
            'recommendations' => $this->generateRecommendations()
        ];
    }

    private function findContradictions() {
        $contradictions = [];
        $allTeachings = [
            $this->micro, $this->innerMicro, $this->innerMeso,
            $this->meso, $this->outerMeso, $this->outerMacro, $this->macro
        ];

        for ($i = 0; $i < count($allTeachings); $i++) {
            for ($j = 0; $j < count($allTeachings); $j++) {
                if ($i != $j) {
                    $teaching1 = $allTeachings[$i];
                    $teaching2 = $allTeachings[$j];

                    $opposite1 = Teaching::getOpposite($teaching1);
                    $opposite2 = Teaching::getOpposite($teaching2);

                    if ($teaching2 == $opposite1 || $teaching1 == $opposite2) {
                        $contradictions[] = $opposite1;
                    }
                }
            }
        }

        return array_unique($contradictions);
    }

    private function calculateCoherenceScore() {
        $teachings = [
            $this->micro, $this->innerMicro, $this->innerMeso,
            $this->meso, $this->outerMeso, $this->outerMacro, $this->macro
        ];

        $uniqueTeachings = count(array_unique($teachings));
        return 1.0 - ($uniqueTeachings - 1) / 6.0;
    }

    private function assessHealingPotential() {
        $emergent = $this->getEmergentProperties();

        if (in_array(Teaching::LOVE, $emergent) && in_array(Teaching::TRUTH, $emergent)) {
            return "HIGH";
        } elseif (in_array(Teaching::LOVE, $emergent) || in_array(Teaching::TRUTH, $emergent)) {
            return "MEDIUM";
        } else {
            return "LOW";
        }
    }

    private function assessScaleIntegration() {
        return [
            'micro_to_meso' => $this->compareScales([$this->micro, $this->innerMicro],
                                                   [$this->meso, $this->outerMeso]),
            'meso_to_macro' => $this->compareScales([$this->meso, $this->outerMeso],
                                                   [$this->macro, $this->outerMacro]),
            'inner_outer_alignment' => $this->compareScales([$this->innerMicro, $this->innerMeso],
                                                           [$this->outerMeso, $this->outerMacro])
        ];
    }

    private function compareScales($scaleA, $scaleB) {
        $common = count(array_intersect($scaleA, $scaleB));
        $total = count(array_unique(array_merge($scaleA, $scaleB)));
        return $total > 0 ? $common / $total : 0.0;
    }

    private function generateRecommendations() {
        $recommendations = [];

        // Calculate metrics locally to avoid infinite recursion
        $coherenceScore = $this->calculateCoherenceScore();
        $contradictions = $this->findContradictions();
        $healingPotential = $this->assessHealingPotential();

        if ($coherenceScore < 0.7) {
            $recommendations[] = "Increase coherence across scales";
        }

        if (!empty($contradictions)) {
            $contradictionList = implode(', ', $contradictions);
            $recommendations[] = "Resolve contradictions: {$contradictionList}";
        }

        if ($healingPotential == 'LOW') {
            $recommendations[] = "Consider integrating LOVE and TRUTH across scales";
        }

        return $recommendations;
    }
}

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Fractal Pattern Analysis</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            padding: 20px;
            background: #f5f5f5;
            margin: 0;
        }
        .fractal-display {
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        h2 {
            color: #333;
            border-bottom: 2px solid #0078d7;
            padding-bottom: 10px;
            margin-top: 0;
        }
        .scales-container {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin: 15px 0;
        }
        .scale {
            display: inline-block;
            padding: 8px 15px;
            background: #0078d7;
            color: white;
            border-radius: 4px;
            font-weight: bold;
            font-size: 14px;
        }
        .analysis-section {
            margin-top: 20px;
            padding: 15px;
            background: #e8f4f8;
            border-left: 4px solid #0078d7;
            border-radius: 4px;
        }
        .analysis-section h3 {
            margin-top: 0;
            color: #0078d7;
        }
        .metric {
            margin: 10px 0;
            padding: 8px;
            background: white;
            border-radius: 4px;
        }
        .metric-label {
            font-weight: bold;
            color: #333;
        }
        .metric-value {
            color: #0078d7;
            font-weight: bold;
        }
        .metric-value.high {
            color: #28a745;
        }
        .metric-value.medium {
            color: #ffc107;
        }
        .metric-value.low {
            color: #dc3545;
        }
        ul {
            margin: 5px 0;
            padding-left: 20px;
        }
        li {
            margin: 5px 0;
        }
    </style>
</head>
<body>
    <div class="fractal-display">
        <?php
        try {
            // Create fractal instance
            $fractal = new PedagogicalFractal(
                $micro,
                $innerMicro,
                $innerMeso,
                $meso,
                $outerMeso,
                $outerMacro,
                $macro
            );

            // Display scales
            echo '<h2>Fractal Scale Pattern</h2>';
            echo '<div class="scales-container">';
            echo '<span class="scale">Micro: ' . htmlspecialchars($fractal->getMicro()) . '</span>';
            echo '<span class="scale">Inner Micro: ' . htmlspecialchars($fractal->getInnerMicro()) . '</span>';
            echo '<span class="scale">Inner Meso: ' . htmlspecialchars($fractal->getInnerMeso()) . '</span>';
            echo '<span class="scale">Meso: ' . htmlspecialchars($fractal->getMeso()) . '</span>';
            echo '<span class="scale">Outer Meso: ' . htmlspecialchars($fractal->getOuterMeso()) . '</span>';
            echo '<span class="scale">Outer Macro: ' . htmlspecialchars($fractal->getOuterMacro()) . '</span>';
            echo '<span class="scale">Macro: ' . htmlspecialchars($fractal->getMacro()) . '</span>';
            echo '</div>';

            // Analyze
            $analysis = $fractal->analyzeFractalHealth();

            echo '<div class="analysis-section">';
            echo '<h3>Health Analysis</h3>';

            echo '<div class="metric">';
            echo '<span class="metric-label">Self-Similar:</span> ';
            echo '<span class="metric-value">' . ($analysis['self_similar'] ? 'Yes' : 'No') . '</span>';
            echo '</div>';

            echo '<div class="metric">';
            echo '<span class="metric-label">Coherence Score:</span> ';
            echo '<span class="metric-value">' . number_format($analysis['coherence_score'], 2) . '</span>';
            echo ' <small>(1.0 = perfect coherence)</small>';
            echo '</div>';

            echo '<div class="metric">';
            echo '<span class="metric-label">Healing Potential:</span> ';
            $potentialClass = strtolower($analysis['healing_potential']);
            echo '<span class="metric-value ' . $potentialClass . '">' . htmlspecialchars($analysis['healing_potential']) . '</span>';
            echo '</div>';

            if (!empty($analysis['emergent_properties'])) {
                echo '<div class="metric">';
                echo '<span class="metric-label">Emergent Properties:</span> ';
                echo '<span class="metric-value">' . implode(', ', $analysis['emergent_properties']) . '</span>';
                echo '</div>';
            }

            if (!empty($analysis['contradictions'])) {
                echo '<div class="metric">';
                echo '<span class="metric-label">Contradictions:</span> ';
                echo '<span class="metric-value low">' . implode(', ', $analysis['contradictions']) . '</span>';
                echo '</div>';
            }

            if (!empty($analysis['recommendations'])) {
                echo '<div class="metric">';
                echo '<span class="metric-label">Recommendations:</span>';
                echo '<ul>';
                foreach ($analysis['recommendations'] as $rec) {
                    echo '<li>' . htmlspecialchars($rec) . '</li>';
                }
                echo '</ul>';
                echo '</div>';
            }

            echo '</div>';

        } catch (Exception $e) {
            echo '<p style="color: #c62828; font-weight: bold;">Error: ' . htmlspecialchars($e->getMessage()) . '</p>';
        }
        ?>
    </div>
</body>
</html>
