# BookGram PHP Embeds

Embed interactive PHP code in your pages with parameter controls.

## Security Model

PHP embeds execute in a **highly restricted environment** with multiple layers of security:

### Security Layers

1. **Static Analysis** - Code is scanned for forbidden functions before execution
2. **Sandboxed Iframe** - Output is displayed in isolated iframe (`sandbox="allow-scripts"`)
3. **Function Blacklist** - 80+ dangerous functions blocked via tokenization
4. **Resource Limits** - 5-second timeout, 32MB memory limit
5. **Directory Isolation** - Files stored in `/php_embeds/`, not directly accessible via HTTP
6. **Path Sanitization** - Directory traversal and protocol handlers blocked
7. **Output Escaping** - All parameters and paths HTML-escaped

## Forbidden Operations

The following are **blocked** and will prevent execution:

### File Operations
- `fopen`, `fwrite`, `fclose`
- `file_get_contents`, `file_put_contents`, `file`
- `readfile`, `fread`, `unlink`, `rmdir`, `mkdir`, `rename`, `copy`
- `tmpfile`, `tempnam`

### Shell Commands
- `exec`, `system`, `shell_exec`, `passthru`
- `popen`, `proc_open`, `proc_close`, `proc_terminate`

### Code Evaluation
- `eval`, `assert`, `create_function`
- `include`, `include_once`, `require`, `require_once`

### Network Operations
- `curl_exec`, `curl_init`, `curl_multi_exec`
- `fsockopen`, `pfsockopen`, `socket_create`, `socket_connect`

### Database Access
- `mysqli_*`, `mysql_*`
- `PDO`, `pg_connect`, `pg_query`
- `sqlite_open`

### PHP Internals
- `dl`, `ini_set`, `ini_alter`, `ini_restore`
- `putenv`, `getenv`
- `set_time_limit`, `ignore_user_abort`

### Sessions & Cookies
- `session_start`, `session_destroy`
- `setcookie`, `setrawcookie`

### Output Control
- `header`, `header_remove`, `headers_sent`, `http_response_code`

### Reflection
- `ReflectionFunction`, `ReflectionClass`, `ReflectionMethod`

### Serialization
- `serialize`, `unserialize`

## Allowed Operations

Safe computational code is allowed:

- **Mathematics** - All math functions and operations
- **String manipulation** - `str_*`, `substr`, `preg_*`, etc.
- **Array operations** - `array_*`, `count`, `sort`, etc.
- **Object-oriented programming** - classes, abstracts, interfaces, inheritance
- **Control structures** - `if`, `for`, `while`, `switch`, `foreach`
- **HTML output** - `echo`, `print`, inline HTML
- **CSS and JavaScript** - Can output styles and scripts in HTML
- **Date/time** - `date()`, `time()`, `strtotime()` (but not `DateTime` with external data)

## Usage

### Basic Embed

Embed a PHP file:

```markdown
[php:fractal_pattern2.php]
```

### With Parameters

Pass parameters to the PHP script:

```markdown
[php:fractal_pattern2.php micro=WISDOM meso=LOVE macro=TRUTH]
```

### With Size

Control iframe dimensions:

```markdown
[php:visualization.php width=900 height=700]
```

### All Together

```markdown
[php:fractal_pattern2.php micro=HUMILITY meso=RESPECT macro=WISDOM width=800 height=600]
```

## Parameter System

Declare interactive parameters in your PHP file header using `BOOKGRAM_PARAM` annotations:

### Parameter Format

```php
// BOOKGRAM_PARAM: name | type | options | default
```

### Parameter Types

#### Select (Dropdown)

```php
// BOOKGRAM_PARAM: color | select | red,blue,green,yellow | blue
// BOOKGRAM_PARAM: teaching | select | WISDOM,LOVE,TRUTH,RESPECT | WISDOM
```

Creates a dropdown menu with the specified options.

#### Range (Slider)

```php
// BOOKGRAM_PARAM: amplitude | range | 0.1-2.0 | 1.0
// BOOKGRAM_PARAM: count | range | 1-100 | 50
```

Creates a slider with min-max range. Automatically detects step size (0.1 for decimals, 1 for integers).

#### Text (Input Field)

```php
// BOOKGRAM_PARAM: label | text | | Default Label
```

Creates a text input field.

### Accessing Parameters

Parameters are automatically available as variables in your code:

```php
<?php
// Parameters are extracted by the executor
$amplitude = $amplitude ?? 1.0;  // Default if not provided
$color = $color ?? 'blue';
$label = $label ?? 'Default Label';
?>
```

## File Structure Template

```php
<?php
/**
 * Your Visualization Title
 * Brief description of what this does
 *
 * BOOKGRAM_PARAM: param1 | select | option1,option2,option3 | option1
 * BOOKGRAM_PARAM: param2 | range | 0-100 | 50
 * BOOKGRAM_PARAM: param3 | text | | Default Text
 */

// Get parameters with defaults
$param1 = $param1 ?? 'option1';
$param2 = $param2 ?? 50;
$param3 = $param3 ?? 'Default Text';

// Your computational logic here
// (No file/network/database operations!)

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Your Visualization</title>
    <style>
        /* Your inline styles */
        body {
            font-family: Arial, sans-serif;
            padding: 20px;
        }
    </style>
</head>
<body>
    <h2><?php echo htmlspecialchars($param3); ?></h2>

    <?php
    // Output your visualization
    echo '<div style="background: ' . htmlspecialchars($param1) . ';">';
    echo 'Value: ' . htmlspecialchars($param2);
    echo '</div>';
    ?>

    <script>
        // Your JavaScript (optional)
        console.log('Visualization loaded');
    </script>
</body>
</html>
```

## Example Files

### fractal_pattern2.php

See `/php_embeds/fractal_pattern2.php` for a complete example with:
- Multiple select parameters (7 teachings)
- Object-oriented code (classes, abstracts, inheritance)
- Complex computation and analysis
- Styled HTML output
- Error handling

This demonstrates a pedagogical fractal pattern analyzer with interactive controls for each scale level.

## Best Practices

### Security

1. **Never** try to read/write files
2. **Never** make external network requests
3. **Never** access databases
4. **Never** execute shell commands
5. **Always** escape output with `htmlspecialchars()`
6. **Always** provide default values for all parameters

### Performance

1. Keep execution time under 5 seconds (hard limit)
2. Avoid infinite loops
3. Limit memory usage (32MB hard limit)
4. Use efficient algorithms

### User Experience

1. Provide meaningful parameter labels
2. Choose appropriate default values
3. Use select dropdowns for discrete choices
4. Use range sliders for continuous values
5. Include helpful descriptions in comments

### Code Quality

1. Use clear, descriptive variable names
2. Add comments to explain complex logic
3. Structure HTML output cleanly
4. Include inline CSS for styling
5. Test with different parameter combinations

## Use Cases

### Educational/Visualization ✓

- Mathematical simulations
- Physics visualizations
- Data structure demonstrations
- Algorithm visualizations
- Pattern analysis
- Statistical calculators
- Geometry renderers

### NOT Allowed ✗

- File upload/download systems
- Database-backed applications
- External API integrations
- Multi-user chat systems
- Session-based applications
- User data collection
- Production web applications

## Troubleshooting

### "Security violation: Forbidden functions detected"

**Problem**: Your code contains blacklisted functions.

**Solution**:
1. Review the forbidden operations list above
2. Remove or replace the prohibited function calls
3. Use only computational operations

Example:
```php
// DON'T DO THIS
$content = file_get_contents('/etc/passwd');  // ✗ Forbidden

// DO THIS INSTEAD
$result = calculateSomething($param1, $param2);  // ✓ Safe computation
```

### "File not found"

**Problem**: PHP file doesn't exist in `/php_embeds/` directory.

**Solution**:
1. Verify file is in `/php_embeds/` directory
2. Check filename spelling in shortcode matches actual file
3. Ensure file extension is `.php`

### "Execution error"

**Problem**: PHP syntax error or runtime exception.

**Solution**:
1. Check PHP syntax in your file
2. Review error logs for details
3. Ensure all parameters have default values
4. Add try-catch blocks around risky operations

### Parameters not updating

**Problem**: Changing parameter controls doesn't update the visualization.

**Solution**:
1. Ensure JavaScript is enabled in browser
2. Check browser console for errors
3. Verify parameter names match BOOKGRAM_PARAM annotations
4. Clear browser cache

## Technical Limitations

- **Execution timeout**: 5 seconds maximum
- **Memory limit**: 32MB maximum
- **Iframe sandbox**: `allow-scripts` only (no same-origin, no forms, no modals)
- **No persistence**: State is not saved between page loads
- **No file access**: Cannot read or write files
- **No network access**: Cannot make HTTP requests

## Support

For issues or questions:
1. Check this README
2. Review `/php_embeds/QUICK_START.md`
3. Examine example files in `/php_embeds/`
4. Review `/php_embeds/.htaccess` for security configuration

## Examples

### Simple Calculator

```php
<?php
/**
 * Simple Calculator
 *
 * BOOKGRAM_PARAM: num1 | range | 0-100 | 10
 * BOOKGRAM_PARAM: num2 | range | 0-100 | 20
 * BOOKGRAM_PARAM: operation | select | add,subtract,multiply,divide | add
 */

$num1 = $num1 ?? 10;
$num2 = $num2 ?? 20;
$operation = $operation ?? 'add';

$operations = [
    'add' => $num1 + $num2,
    'subtract' => $num1 - $num2,
    'multiply' => $num1 * $num2,
    'divide' => $num2 != 0 ? $num1 / $num2 : 'Error: Division by zero'
];

$result = $operations[$operation] ?? 'Unknown operation';
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Calculator</title>
    <style>
        body { font-family: Arial, sans-serif; padding: 20px; }
        .result { font-size: 24px; font-weight: bold; color: #0078d7; }
    </style>
</head>
<body>
    <h2>Calculator Result</h2>
    <p class="result">
        <?php echo htmlspecialchars($num1); ?>
        <?php echo htmlspecialchars($operation); ?>
        <?php echo htmlspecialchars($num2); ?>
        = <?php echo htmlspecialchars($result); ?>
    </p>
</body>
</html>
```

### Pattern Generator

```php
<?php
/**
 * Pattern Generator
 *
 * BOOKGRAM_PARAM: size | range | 5-50 | 20
 * BOOKGRAM_PARAM: color | select | red,blue,green,purple | blue
 */

$size = $size ?? 20;
$color = $color ?? 'blue';
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Pattern</title>
</head>
<body>
    <svg width="<?php echo $size * 10; ?>" height="<?php echo $size * 10; ?>">
        <?php
        for ($i = 0; $i < $size; $i++) {
            echo '<circle cx="' . ($i * 10 + 5) . '" cy="' . ($i * 10 + 5) . '" ';
            echo 'r="5" fill="' . htmlspecialchars($color) . '" />';
        }
        ?>
    </svg>
</body>
</html>
```

---

**Remember**: PHP embeds are designed for **educational and visualization purposes only**. They are not suitable for production web applications or systems that require file storage, databases, or external API access.
