# PHP Embeds Quick Start

Get started with interactive PHP visualizations in 3 steps!

## Step 1: Create PHP File

Create a file in `/php_embeds/` directory. For example, `my_visualization.php`:

```php
<?php
/**
 * My First Visualization
 *
 * BOOKGRAM_PARAM: size | range | 10-100 | 50
 * BOOKGRAM_PARAM: color | select | red,blue,green,purple | blue
 */

// Get parameters with defaults
$size = $size ?? 50;
$color = $color ?? 'blue';
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>My Visualization</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            padding: 20px;
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100vh;
            margin: 0;
        }
    </style>
</head>
<body>
    <div style="
        width: <?php echo $size; ?>px;
        height: <?php echo $size; ?>px;
        background: <?php echo $color; ?>;
        border: 2px solid black;
        border-radius: 8px;
        display: flex;
        justify-content: center;
        align-items: center;
        color: white;
        font-weight: bold;
    ">
        Size: <?php echo $size; ?>px
    </div>
</body>
</html>
```

## Step 2: Embed in Page

In your BookGram page content, use the `[php:...]` shortcode:

```markdown
# My Page

Here's my interactive visualization:

[php:my_visualization.php]

Try changing the parameters using the controls above!

You can also set default values:

[php:my_visualization.php size=75 color=green]
```

## Step 3: View and Interact

Visit your page! You'll see:

1. **Parameter controls** above the visualization
2. **Interactive sliders and dropdowns** to adjust values
3. **Live updates** when you change parameters
4. **Sandboxed iframe** containing your visualization

## How Parameters Work

### Define Parameters

Use `BOOKGRAM_PARAM` comments in your PHP file header:

```php
// BOOKGRAM_PARAM: name | type | options | default
```

### Parameter Types

**Select (Dropdown)**:
```php
// BOOKGRAM_PARAM: shape | select | circle,square,triangle | circle
```

**Range (Slider)**:
```php
// BOOKGRAM_PARAM: radius | range | 1-100 | 50
```

**Text (Input)**:
```php
// BOOKGRAM_PARAM: title | text | | My Title
```

### Access in Code

Parameters are automatically available:

```php
<?php
$shape = $shape ?? 'circle';   // Fallback to default
$radius = $radius ?? 50;
$title = $title ?? 'My Title';
?>
```

## Complete Examples

### Example 1: Simple Calculator

**File**: `/php_embeds/calculator.php`

```php
<?php
/**
 * BOOKGRAM_PARAM: a | range | 0-100 | 10
 * BOOKGRAM_PARAM: b | range | 0-100 | 5
 * BOOKGRAM_PARAM: op | select | +,-,*,/ | +
 */
$a = $a ?? 10;
$b = $b ?? 5;
$op = $op ?? '+';

$result = match($op) {
    '+' => $a + $b,
    '-' => $a - $b,
    '*' => $a * $b,
    '/' => $b != 0 ? $a / $b : 'Error',
    default => 'Unknown'
};
?>
<!DOCTYPE html>
<html>
<head><meta charset="UTF-8"><title>Calculator</title></head>
<body style="font-family: Arial; padding: 30px; text-align: center;">
    <h2><?php echo "$a $op $b = $result"; ?></h2>
</body>
</html>
```

**Usage**:
```markdown
[php:calculator.php]
```

### Example 2: Pattern Grid

**File**: `/php_embeds/grid.php`

```php
<?php
/**
 * BOOKGRAM_PARAM: rows | range | 1-20 | 5
 * BOOKGRAM_PARAM: cols | range | 1-20 | 5
 * BOOKGRAM_PARAM: color | select | red,blue,green,orange | blue
 */
$rows = $rows ?? 5;
$cols = $cols ?? 5;
$color = $color ?? 'blue';
?>
<!DOCTYPE html>
<html>
<head><meta charset="UTF-8"><title>Grid</title></head>
<body style="font-family: Arial; padding: 20px;">
    <div style="display: grid; grid-template-columns: repeat(<?php echo $cols; ?>, 50px); gap: 5px;">
        <?php
        for ($i = 0; $i < $rows * $cols; $i++) {
            echo '<div style="width: 50px; height: 50px; background: ' . htmlspecialchars($color) . '; border-radius: 4px;"></div>';
        }
        ?>
    </div>
</body>
</html>
```

**Usage**:
```markdown
[php:grid.php rows=10 cols=10 color=green width=600]
```

### Example 3: Text Analyzer

**File**: `/php_embeds/analyzer.php`

```php
<?php
/**
 * BOOKGRAM_PARAM: text | text | | Hello World
 */
$text = $text ?? 'Hello World';
$length = strlen($text);
$words = str_word_count($text);
$upper = strtoupper($text);
$reverse = strrev($text);
?>
<!DOCTYPE html>
<html>
<head><meta charset="UTF-8"><title>Analyzer</title>
<style>
    body { font-family: Arial; padding: 20px; }
    .stat { padding: 10px; margin: 5px; background: #e8f4f8; border-left: 4px solid #0078d7; }
</style>
</head>
<body>
    <h2>Text Analysis</h2>
    <div class="stat">Length: <?php echo $length; ?> characters</div>
    <div class="stat">Words: <?php echo $words; ?></div>
    <div class="stat">Uppercase: <?php echo htmlspecialchars($upper); ?></div>
    <div class="stat">Reversed: <?php echo htmlspecialchars($reverse); ?></div>
</body>
</html>
```

**Usage**:
```markdown
[php:analyzer.php text="BookGram is awesome"]
```

## Common Patterns

### Always Escape Output

```php
// GOOD - Safe
echo htmlspecialchars($user_input, ENT_QUOTES);

// BAD - Unsafe
echo $user_input;
```

### Provide Defaults

```php
// GOOD - Has fallback
$value = $value ?? 'default';

// BAD - May be undefined
echo $value;
```

### Validate Input

```php
// GOOD - Validated
$count = max(1, min(100, $count ?? 10));

// OK - Has default
$count = $count ?? 10;
```

## Tips

1. **Start simple** - Begin with basic examples, add complexity gradually
2. **Use inline styles** - Easier than external CSS files
3. **Test parameters** - Try different combinations
4. **Check examples** - Review `/php_embeds/fractal_pattern2.php` for advanced patterns
5. **Read README** - Full documentation in `/php_embeds/README.md`

## What's Allowed

✓ Math calculations
✓ String manipulation
✓ Array operations
✓ HTML/CSS/JavaScript output
✓ Object-oriented code
✓ Control structures

## What's NOT Allowed

✗ File operations
✗ Database queries
✗ Network requests
✗ Shell commands
✗ Session management
✗ External APIs

## Troubleshooting

**Controls not showing?**
- Check `BOOKGRAM_PARAM` syntax (must be in comments)
- Ensure format: `name | type | options | default`

**Security error?**
- Remove forbidden functions (file_get_contents, exec, etc.)
- See README for complete blacklist

**Not updating?**
- Enable JavaScript in browser
- Check browser console for errors

**Timeout error?**
- Simplify computation
- Avoid infinite loops
- Stay under 5-second limit

## Next Steps

1. Create your first visualization
2. Experiment with parameter types
3. Study `/php_embeds/fractal_pattern2.php` for advanced example
4. Read full `/php_embeds/README.md` for security details
5. Build something amazing!

---

**Remember**: Keep it simple, keep it safe, and have fun visualizing!
