<?php
// index.php

require_once 'app/core/auth.php'; // Handles sessions and auth functions
require_once 'app/core/functions.php'; // Handles data/content logic
require_once 'app/core/Parsedown.php'; // Handles Markdown parsing

// Load site-wide data
$config = load_config();

// Simple Routing
$request_uri = $_SERVER['REQUEST_URI'] ?? '/';
$request_path = parse_url($request_uri, PHP_URL_PATH);
$parts = explode('/', trim($request_path, '/'));
$page_slug = $parts[0] ?? 'home';

// --- Path-based Routing ---
switch ($page_slug) {
    case 'home':
    case '':
        // --- BROWSER ---
        $data['current_path'] = sanitize_path($_GET['path'] ?? '');
        $data['contents'] = get_directory_contents($data['current_path']);
        load_template('browser', $data);
        break;

    case 'page':
        // --- VIEW A PAGE ---
        $path = implode('/', array_slice($parts, 1));
        // Use increment function
        if (empty($path) || !$data['page'] = increment_and_get_page_data($path)) {
            show_404();
        }

        // Check if user can view this private page
        if (!can_view_private_page($data['page'])) {
            // Check if this is a password-protected page with valid token
            if (!empty($data['page']['is_private']) &&
                !empty($data['page']['password_hash']) &&
                isset($_GET['token']) &&
                !empty($data['page']['share_token']) &&
                hash_equals($data['page']['share_token'], $_GET['token'])) {
                // Show password prompt
                $data['path'] = $path;
                $data['error_message'] = '';
                load_template('password-prompt', $data);
                exit;
            }

            http_response_code(403);
            die('<h1>403 Forbidden</h1><p>This page is private. You need a valid share link to access it.</p>');
        }

        $data['revision_id_to_load'] = $data['page']['current_revision_id'] ?? null; // Handle missing key
        $data['is_old_revision'] = false;
        $data['history'] = $data['page']['history'] ?? []; // Ensure history is an array

        // Check against stringified history array elements for safety
        if (isset($_GET['rev']) && is_array($data['history']) && in_array((string)$_GET['rev'], array_map('strval', $data['history']))) {
            $data['revision_id_to_load'] = (string)$_GET['rev']; // Store as string
            $data['is_old_revision'] = ($data['revision_id_to_load'] != ($data['page']['current_revision_id'] ?? null));
        }

        // Check if revision ID exists before getting content
        if ($data['revision_id_to_load'] === null) {
             error_log("Error: No current revision ID found for page: " . $path);
             $data['content'] = 'Error: Page data is incomplete (missing revision ID).';
        } else {
             $data['content'] = get_revision_content($path, $data['revision_id_to_load']);
        }

        load_template('page', $data);
        break;

    case 'edit':
        // --- EDIT A PAGE ---
        require_auth('editor');
        $path = implode('/', array_slice($parts, 1));

        if (strpos($path, 'new-page-in-') === 0) {
            $path = str_replace('new-page-in-', '', $path);
            $data['current_path'] = ($path === 'root') ? '' : $path;
            $data['page'] = null; // Indicate it's a new page
            $data['content'] = "# New Page\n\nStart writing.";
        } else {
            $data['page'] = get_page_data($path); // Get existing data (includes UUID)
            if ($data['page']) {
                $current_rev_id = $data['page']['current_revision_id'] ?? null;
                if ($current_rev_id) {
                    $data['content'] = get_revision_content($path, $current_rev_id);
                } else {
                     $data['content'] = 'Error: Could not load current revision content.';
                     error_log("Edit error: Missing current_revision_id for page " . $path);
                }
                $data['current_path'] = dirname($path) === '.' ? '' : dirname($path);
            } else {
                // If page doesn't exist (e.g., direct edit URL), treat as new in that path
                $data['current_path'] = dirname($path) === '.' ? '' : dirname($path); // Put in parent dir
                $data['page'] = null; // Indicate new page
                $data['content'] = "# New Page\n\nStart writing.";
            }
        }
        load_template('edit', $data);
        break;


    case 'save':
        // --- ACTION: SAVE CONTENT ---
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { die('Invalid request.'); }
        require_auth('editor');
        validate_csrf_token($_POST['csrf_token'] ?? '');

        $current_path = sanitize_path($_POST['path'] ?? '');
        $title = trim($_POST['title'] ?? 'Untitled');
        $content = $_POST['content'] ?? '';
        if (empty($title)) $title = 'Untitled';

        // --- Handle Image Upload ---
        $final_image_url = $_POST['feature_image_url'] ?? '';

        $new_image_path = handle_image_upload('feature_image_file', 'features');
        if ($new_image_path !== null) {
            $final_image_url = $new_image_path;
        }
        // --- End Upload Logic ---

        // Get or create the slug
        $is_existing_page = isset($_POST['is_existing_page']) && $_POST['is_existing_page'] === '1';

        if ($is_existing_page) {
            // For existing pages, ALWAYS use the original slug to prevent creating duplicates
            $slug = sanitize_path($_POST['original_slug'] ?? $_POST['slug'] ?? '');
            if (empty($slug)) {
                die("Error: Cannot save existing page without a slug. Please contact administrator.");
            }
        } else {
            // For new pages, generate slug from title if not provided
            $slug = sanitize_path($_POST['slug'] ?? '');
            if (empty($slug)) {
                $slug = slugify($title);
            }
            if (empty($slug) || $slug === 'n-a') {
                $slug = 'page-' . time();
            }
        }

        $full_path = ($current_path ? $current_path . '/' : '') . $slug;

        // Get privacy setting
        $is_private = isset($_POST['is_private']) && $_POST['is_private'] === '1';

        // Get password setting
        $password = null;
        if ($is_private) {
            if (isset($_POST['remove_password']) && $_POST['remove_password'] === '1') {
                // Explicitly remove password
                $password = '';
            } elseif (isset($_POST['page_password']) && trim($_POST['page_password']) !== '') {
                // Set new password
                $password = trim($_POST['page_password']);
            }
            // If password is null, it will preserve existing password
        }

        if (save_page_content($full_path, $title, $content, $_SESSION['user'], $final_image_url, $is_private, $password)) {
            $_SESSION['admin_message'] = $is_existing_page ? 'Page updated successfully!' : 'Page created successfully!';
            header('Location: /page/' . $full_path);
        } else {
            die("Error: Could not save the page. Please check server logs and permissions.");
        }
        exit;

    case 'verify-password':
        // --- ACTION: VERIFY PASSWORD FOR PRIVATE PAGE ---
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { die('Invalid request.'); }
        validate_csrf_token($_POST['csrf_token'] ?? '');

        $page_path = sanitize_path($_POST['page_path'] ?? '');
        $password = $_POST['password'] ?? '';
        $token = $_POST['token'] ?? '';

        if (empty($page_path)) {
            die('Invalid page path.');
        }

        $page_data = get_page_data($page_path);
        if (!$page_data) {
            show_404();
        }

        // Verify token is valid
        if (empty($token) || empty($page_data['share_token']) || !hash_equals($page_data['share_token'], $token)) {
            http_response_code(403);
            die('<h1>403 Forbidden</h1><p>Invalid share token.</p>');
        }

        // Verify password
        if (verify_page_password($page_data, $password)) {
            // Password correct - redirect to page with token
            header('Location: /page/' . $page_path . '?token=' . urlencode($token));
            exit;
        } else {
            // Password incorrect - show prompt again with error
            $data['page'] = $page_data;
            $data['path'] = $page_path;
            $data['error_message'] = 'Incorrect password. Please try again.';
            load_template('password-prompt', $data);
            exit;
        }

    case 'plugin-admin':
        // --- PLUGIN ADMIN PROXY ---
        require_auth('editor');

        // Parse URL: /plugin-admin/{page_uuid}/{plugin_id}/{instance_id}
        $page_uuid = $parts[1] ?? null;
        $plugin_id = $parts[2] ?? null;
        $instance_id = $parts[3] ?? null;

        if (!$page_uuid || !$plugin_id || !$instance_id) {
            die('Invalid plugin admin URL. Expected format: /plugin-admin/{page_uuid}/{plugin_id}/{instance_id}');
        }

        // Load plugin system
        require_once 'app/core/plugins.php';

        // Create instance if it doesn't exist yet (lazy initialization)
        if (!instance_exists($page_uuid, $plugin_id, $instance_id)) {
            // Auto-create instance when accessing admin for the first time
            $instance_created = create_plugin_instance($plugin_id, $instance_id, $page_uuid);
            if (!$instance_created) {
                die('Failed to create plugin instance. Check permissions on app/storage/plugin_instances/');
            }
        }

        // Create auth token for plugin
        $auth_token = create_plugin_auth_token($page_uuid, $plugin_id, $instance_id);

        if (!$auth_token) {
            die('Failed to create authentication token. Please try again.');
        }

        // Get plugin manifest to determine admin entry point
        $plugins = discover_plugins();
        $plugin_manifest = null;
        foreach ($plugins as $plugin) {
            if ($plugin['id'] === $plugin_id) {
                $plugin_manifest = $plugin['manifest'];
                break;
            }
        }

        if (!$plugin_manifest) {
            die('Plugin not found: ' . htmlspecialchars($plugin_id));
        }

        // Build plugin admin URL with token using manifest's admin_entry
        $admin_entry = $plugin_manifest['admin_entry'] ?? 'admin/index.php';
        $plugin_admin_path = $plugin_id . '/' . $admin_entry;
        $plugin_admin_url = '/' . $plugin_admin_path . '?bga_token=' . urlencode($auth_token) . '&instance=' . urlencode($instance_id);

        // Redirect to plugin admin with token
        header('Location: ' . $plugin_admin_url);
        exit;

    case 'php-execute':
        // --- EXECUTE PHP EMBED ---
        require_once 'app/core/php_executor.php';
        execute_php_embed_secure();
        exit;

    case 'page-image':
        // --- SERVE PAGE IMAGE ---
        // URL format: /page-image/{page-uuid}/{filename}
        $page_uuid = $parts[1] ?? null;
        $filename = $parts[2] ?? null;

        if (!$page_uuid || !$filename) {
            http_response_code(404);
            die('Invalid image URL');
        }

        require_once 'app/core/functions.php';
        serve_page_image($page_uuid, $filename);
        break;

    case 'admin':
        // --- ADMIN PANEL ---
        require_auth('admin');
        $data['all_users'] = load_users();
        load_template('admin', $data);
        break;

    case 'admin-action':
        // --- ACTION: ADMIN ACTIONS ---
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { die('Invalid request.'); }

        $action = $_POST['action'] ?? '';

        // DEBUG: Log the action being attempted
        $debug_log = __DIR__ . '/debug.log';
        file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Admin action requested: " . $action . "\n", FILE_APPEND);

        // Page image actions require editor role, other actions require admin role
        $page_image_actions = ['upload_page_image', 'delete_page_image', 'update_page_image'];
        if (in_array($action, $page_image_actions)) {
            file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Page image action detected, requiring editor auth\n", FILE_APPEND);
            require_auth('editor'); // Editors can manage images on pages they can edit
        } else {
            require_auth('admin'); // Keep all other actions admin-only
        }

        validate_csrf_token($_POST['csrf_token'] ?? '');
        $current_path = sanitize_path($_POST['path'] ?? ''); // Path where the action was initiated
        $redirect_url = '/?' . ($current_path ? 'path=' . $current_path : ''); // Default redirect back

        $success = false; // Track action success

        if ($action === 'create-folder') {
            $success = create_folder($current_path, $_POST['folder_name']);
        }

        if ($action === 'delete-item') {
            $item_path = sanitize_path($_POST['item_path']);
            if (is_dir(CONTENT_DIR . $item_path)) {
                $success = delete_folder_recursive($item_path);
            } else {
                $success = delete_page($item_path);
            }
        }

        if ($action === 'rename-item') {
            $success = rename_item(
                sanitize_path($_POST['item_path']),
                $_POST['new_name']
            );
        }

        if ($action === 'move-item') {
            $success = move_item(
                sanitize_path($_POST['item_path']), // The item being moved
                sanitize_path($_POST['target_path']) // The destination folder path
            );
            if ($success) {
                $redirect_url = '/?' . ($_POST['target_path'] ? 'path=' . sanitize_path($_POST['target_path']) : '');
            }
        }

        if ($action === 'delete_revision') {
            $page_path = sanitize_path($_POST['path']); // Get page path for redirect
            $success = delete_revision($page_path, $_POST['revision_id']);
            $redirect_url = '/page/' . $page_path . ($success ? '' : '?error=revdelete');
            header('Location: ' . $redirect_url);
            exit;
        }

        if ($action === 'revert-revision') {
            $page_path = sanitize_path($_POST['path']); // Get page path
            $success = revert_to_revision(
                $page_path,
                $_POST['revision_id'],
                $_SESSION['user'] // Pass the admin user performing the action
            );
            $redirect_url = '/page/' . $page_path . ($success ? '?reverted=1' : '?error=revertfail');
            header('Location: ' . $redirect_url); // Redirect immediately after revert attempt
            exit;
        }


        if ($action === 'save_settings') {
            $new_config = [
                'site_title' => $_POST['site_title'] ?? $config['site_title'],
                'site_description' => $_POST['site_description'] ?? $config['site_description'],
                'site_logo' => $config['site_logo'] ?? '/public/uploads/logo.png', // Ensure default
                'footer_prefix' => $_POST['footer_prefix'] ?? $config['footer_prefix'] ?? '' // Ensure default
            ];
            $success = save_config($new_config);
            $redirect_url = '/admin'; // Stay on admin page for settings
        }

        if ($action === 'chatgram_settings') {
            $config_data = load_config();
            $config_data['chatgram'] = [
                'enabled' => isset($_POST['chatgram_enabled']),
                'default_access_mode' => $_POST['default_access_mode'] ?? 'login_only',
                'message_retention_days' => intval($_POST['retention_days'] ?? 30),
                'rate_limit_messages_per_minute' => intval($_POST['rate_limit'] ?? 10),
                'guest_timeout_minutes' => 10,
                'max_message_length' => 1000,
                'allow_private_messaging' => true
            ];
            $success = save_config($config_data);
            $redirect_url = '/admin';
        }

        if ($action === 'add_user') {
            $success = add_user( // Assuming add_user returns true/false
                $_POST['username'],
                $_POST['password'],
                $_POST['role'],
                $_SESSION['user']
            );
            $redirect_url = '/admin'; // Stay on admin page
        }

        if ($action === 'delete_user') {
            $success = delete_user($_POST['username']); // Assuming delete_user returns true/false
            $redirect_url = '/admin'; // Stay on admin page
        }

        if ($action === 'upload_html_embed') {
            $uploaded_file = handle_html_upload('html_file');
            if ($uploaded_file !== null) {
                $_SESSION['admin_message'] = 'HTML embed uploaded successfully: ' . $uploaded_file;
                $success = true;
            } else {
                $_SESSION['admin_error'] = 'Failed to upload HTML embed. Check file type and size.';
                $success = false;
            }
            $redirect_url = '/admin';
        }

        if ($action === 'delete_html_embed') {
            $filename = $_POST['filename'] ?? '';
            if (delete_html_embed($filename)) {
                $_SESSION['admin_message'] = 'Embed deleted successfully: ' . $filename;
                $success = true;
            } else {
                $_SESSION['admin_error'] = 'Failed to delete embed. File may not exist or is outside embeds directory.';
                $success = false;
            }
            $redirect_url = '/admin';
        }

        if ($action === 'upload_php_embed') {
            $uploaded_file = handle_php_upload('php_file');
            if ($uploaded_file !== null) {
                $_SESSION['admin_message'] = 'PHP embed uploaded successfully: ' . $uploaded_file;
                $success = true;
            } else {
                // Error message already set by handle_php_upload if not already set
                if (!isset($_SESSION['admin_error'])) {
                    $_SESSION['admin_error'] = 'Failed to upload PHP file. Check file type and size.';
                }
                $success = false;
            }
            $redirect_url = '/admin';
        }

        if ($action === 'delete_php_embed') {
            $filename = $_POST['filename'] ?? '';
            if (delete_php_embed($filename)) {
                $_SESSION['admin_message'] = 'PHP embed deleted successfully: ' . $filename;
                $success = true;
            } else {
                $_SESSION['admin_error'] = 'Failed to delete PHP embed. File may not exist or is outside php_embeds directory.';
                $success = false;
            }
            $redirect_url = '/admin';
        }

        if ($action === 'upload_page_image') {
            $page_uuid = $_POST['page_uuid'] ?? '';
            file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Processing upload_page_image for UUID: $page_uuid\n", FILE_APPEND);

            if (empty($page_uuid)) {
                file_put_contents($debug_log, date('Y-m-d H:i:s') . " - ERROR: Invalid page UUID\n", FILE_APPEND);
                $_SESSION['admin_error'] = 'Invalid page UUID';
                $success = false;
            } else {
                $image_metadata = handle_page_image_upload($page_uuid, 'page_image_file');

                if ($image_metadata) {
                    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Image metadata received, saving...\n", FILE_APPEND);
                    // Add to metadata array
                    $images = load_page_images($page_uuid);
                    $images[] = $image_metadata;

                    if (save_page_images_metadata($page_uuid, $images)) {
                        file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Metadata saved successfully. Total images: " . count($images) . "\n", FILE_APPEND);
                        $_SESSION['admin_message'] = 'Image uploaded successfully (' . count($images) . ' total images)';
                        $success = true;
                    } else {
                        file_put_contents($debug_log, date('Y-m-d H:i:s') . " - ERROR: Failed to save image metadata\n", FILE_APPEND);
                        $_SESSION['admin_error'] = 'Failed to save image metadata';
                        $success = false;
                    }
                } else {
                    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - ERROR: handle_page_image_upload returned null\n", FILE_APPEND);
                    $_SESSION['admin_error'] = 'Failed to upload image. Check file type and size.';
                    $success = false;
                }
            }

            // Redirect back to edit page
            $redirect_path = sanitize_path($_POST['redirect_path'] ?? '');

            if (empty($redirect_path)) {
                // Fallback: try to find page path from UUID if redirect_path is missing
                file_put_contents($debug_log, date('Y-m-d H:i:s') . " - WARNING: redirect_path is empty, attempting to find page from UUID\n", FILE_APPEND);
                $uuid_index = load_uuid_index();
                if (isset($uuid_index[$page_uuid])) {
                    $redirect_path = $uuid_index[$page_uuid];
                    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Found page path from UUID: $redirect_path\n", FILE_APPEND);
                } else {
                    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - ERROR: Could not find page path for UUID: $page_uuid\n", FILE_APPEND);
                    $_SESSION['admin_error'] = 'Upload may have succeeded but could not redirect. Check your pages.';
                    $redirect_path = ''; // Will redirect to home
                }
            }

            $redirect_url = $redirect_path ? '/edit/' . $redirect_path : '/?error=redirect';
            file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Redirecting to: $redirect_url\n", FILE_APPEND);
        }

        if ($action === 'delete_page_image') {
            $page_uuid = $_POST['page_uuid'] ?? '';
            $image_id = $_POST['image_id'] ?? '';

            if (delete_page_image($page_uuid, $image_id)) {
                $_SESSION['admin_message'] = 'Image deleted successfully';
                $success = true;
            } else {
                $_SESSION['admin_error'] = 'Failed to delete image';
                $success = false;
            }

            // Redirect back to edit page
            $redirect_path = sanitize_path($_POST['redirect_path'] ?? '');
            if (empty($redirect_path)) {
                $uuid_index = load_uuid_index();
                $redirect_path = $uuid_index[$page_uuid] ?? '';
            }
            $redirect_url = $redirect_path ? '/edit/' . $redirect_path : '/?error=redirect';
        }

        if ($action === 'update_page_image') {
            $page_uuid = $_POST['page_uuid'] ?? '';
            $image_id = $_POST['image_id'] ?? '';
            $description = $_POST['description'] ?? '';

            if (update_page_image_metadata($page_uuid, $image_id, $description)) {
                $_SESSION['admin_message'] = 'Image updated successfully';
                $success = true;
            } else {
                $_SESSION['admin_error'] = 'Failed to update image';
                $success = false;
            }

            // Redirect back to edit page
            $redirect_path = sanitize_path($_POST['redirect_path'] ?? '');
            if (empty($redirect_path)) {
                $uuid_index = load_uuid_index();
                $redirect_path = $uuid_index[$page_uuid] ?? '';
            }
            $redirect_url = $redirect_path ? '/edit/' . $redirect_path : '/?error=redirect';
        }

        // Add error handling feedback if needed (e.g., using session flash messages)
        // if (!$success && !in_array($action, ['delete_revision', 'revert-revision'])) { // Avoid double messaging
        //     $_SESSION['flash_error'] = "Action '$action' failed. Please check logs.";
        // }

        header('Location: ' . $redirect_url);
        exit;

    case 'rss':
        // --- RSS FEED ---
        $data['pages'] = array_slice(get_all_pages_sorted_by_date(), 0, 20); // Get latest 20
        $data['parser'] = new Parsedown(); // Pass the parser
        $data['base_url'] = (isset($_SERVER['HTTPS']) ? "https" : "http") . "://" . ($_SERVER['HTTP_HOST'] ?? 'localhost');
        load_template_rss('rss', $data); // Use the RSS loader (includes exit)
        break;

    case 'search':
        // --- SEARCH ---
        $data['query'] = $_GET['q'] ?? '';
        $data['results'] = [];
        $data['hashtag'] = null;

        if (!empty($data['query'])) {
            // Check if search is for a hashtag
            if (strpos($data['query'], '#') === 0) {
                $hashtag = substr($data['query'], 1);
                $data['hashtag'] = $hashtag;
                $data['results'] = get_pages_by_hashtag($hashtag);
            } else {
                $data['results'] = search_pages($data['query']);
            }
        }
        load_template('search', $data);
        break;

    case 'login':
        // --- LOGIN ---
        $data['error'] = null;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $remember = isset($_POST['remember_me']) && $_POST['remember_me'] === '1';
            if (check_login($_POST['username'], $_POST['password'], $remember)) {
                // Redirect to the page they were trying to access, or referer, or home
                $redirect_url = $_POST['redirect'] ?? $_SERVER['HTTP_REFERER'] ?? '/';
                // Sanitize redirect to prevent open redirect attacks
                $parsed = parse_url($redirect_url);
                $current_host = $_SERVER['HTTP_HOST'] ?? '';
                if (isset($parsed['host']) && $parsed['host'] !== $current_host) {
                    $redirect_url = '/'; // Only allow local redirects
                }
                // Don't redirect back to login page
                if (strpos($redirect_url, '/login') !== false) {
                    $redirect_url = '/';
                }
                header('Location: ' . $redirect_url);
                exit;
            } else {
                $data['error'] = 'Invalid username or password.';
            }
        }
        // Store current page to redirect back after login
        $data['redirect'] = $_GET['redirect'] ?? $_SERVER['HTTP_REFERER'] ?? '';
        load_template('login', $data);
        break;

    case 'logout':
        // --- LOGOUT ---
        logout();
        header('Location: /');
        exit;

    case 'profile':
        // --- VIEW PROFILE ---
        $username = $parts[1] ?? null;
        $all_users = load_users();

        if (!$username || !isset($all_users[$username])) {
            show_404();
        }

        $data['profile_user'] = $all_users[$username];
        $data['pages'] = get_pages_by_author($username); // Get pages by this author
        load_template('profile', $data);
        break;

    case 'profile-edit':
        // --- EDIT OWN PROFILE ---
        require_auth('editor'); // Must be logged in
        $all_users = load_users();
        if (!isset($_SESSION['user']) || !isset($all_users[$_SESSION['user']])) {
            logout(); // Log out if user data is missing or session invalid
            header('Location: /login');
            exit;
        }
        $data['profile_user'] = $all_users[$_SESSION['user']];
        load_template('profile-edit', $data);
        break;

    case 'profile-save':
        // --- SAVE OWN PROFILE ---
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') { die('Invalid request.'); }
        require_auth('editor'); // Must be logged in
        validate_csrf_token($_POST['csrf_token'] ?? '');

        $logged_in_user = $_SESSION['user'];

        // --- Handle Profile Photo Upload ---
        $all_users = load_users();
        if (!isset($all_users[$logged_in_user])) {
             die("Error: User data not found.");
        }
        $current_photo_url = $all_users[$logged_in_user]['photo_url'] ?? '';
        $final_photo_url = $current_photo_url;

        $new_image_path = handle_image_upload('profile_photo', 'avatars');
        if ($new_image_path !== null) {
            $final_photo_url = $new_image_path;
        }
        // --- End Upload Logic ---

        // --- Save basic profile info ---
        $profile_update_success = update_user_profile(
            $logged_in_user,
            $_POST['tagline'] ?? '',
            $final_photo_url,
            $_POST['bio'] ?? ''
        );

        // --- Handle Password Change ---
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        $password_update_success = true; // Assume success if not attempting change
        $password_error = null;

        if (!empty($new_password)) {
            if ($new_password === $confirm_password) {
                $password_update_success = update_user_password($logged_in_user, $new_password);
                if (!$password_update_success) {
                    $password_error = "Failed to update password. Please try again.";
                    error_log("Password update failed for user: " . $logged_in_user);
                }
            } else {
                $password_update_success = false;
                $password_error = "New passwords do not match.";
            }
        }
        // --- END Password Change ---

        if ($profile_update_success && $password_update_success) {
            // Optional: $_SESSION['flash_success'] = "Profile updated successfully.";
            header('Location: /profile/' . $logged_in_user);
        } else {
            $error_message = !$profile_update_success ? "Failed to update profile info. " : "";
            $error_message .= $password_error ?? "";
            error_log("Profile save failed for user: " . $logged_in_user . " Errors: " . trim($error_message));
            // Optional: Use flash messages: $_SESSION['flash_error'] = trim($error_message);
             header('Location: /profile-edit'); // Redirect back to edit form
        }
        exit;

    case 'about':
        // --- ABOUT PAGE ---
        load_template('about');
        break;

    case 'site-tree':
        // --- SITE TREE PAGE ---
        load_template('site-tree');
        break;

    case 'api':
        // --- API ENDPOINTS ---
        $api_action = $parts[1] ?? '';

        if ($api_action === 'hashtag-autocomplete') {
            header('Content-Type: application/json');
            $prefix = $_GET['q'] ?? '';
            if (empty($prefix)) {
                echo json_encode([]);
            } else {
                $suggestions = search_hashtags($prefix, 10);
                echo json_encode($suggestions);
            }
            exit;
        }

        if ($api_action === 'top-hashtags') {
            header('Content-Type: application/json');
            $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 100;
            $top_tags = get_top_hashtags($limit);
            echo json_encode($top_tags);
            exit;
        }

        http_response_code(404);
        echo json_encode(['error' => 'API endpoint not found']);
        exit;

    default:
        // --- 404 NOT FOUND or Direct Page Access ---
        $path = implode('/', $parts);
        $page_data_check = get_page_data($path); // Check existence

        if ($page_data_check) {
            // Now load with hit increment
            $data['page'] = increment_and_get_page_data($path);
            if ($data['page']) {
                $data['revision_id_to_load'] = $data['page']['current_revision_id'] ?? null;
                $data['is_old_revision'] = false; // Direct access always loads current
                 $data['history'] = $data['page']['history'] ?? [];

                if ($data['revision_id_to_load']) {
                     $data['content'] = get_revision_content($path, $data['revision_id_to_load']);
                     load_template('page', $data);
                     exit; // Found and loaded page
                } else {
                     error_log("Default route error: Missing revision ID for page " . $path);
                     // Fall through to 404 if data is incomplete
                }
            } else {
                 error_log("Default route error: increment_and_get_page_data failed for " . $path);
                 // Fall through to 404
            }
        }

        // If not found as a page, show 404
        show_404();
}

// Function to display 404 page
function show_404() {
    if (!headers_sent()) {
        http_response_code(404);
    }
    load_template('404');
    exit;
}