# HTML Embeds Directory

This directory contains HTML files that can be embedded into BookGram pages using the `[html:filename.html]` shortcode.

## Usage

1. Place your HTML files in this directory (or subdirectories)
2. Use the shortcode in your page content:
   ```
   [html:myfile.html]
   [html:demos/chart.html width=800 height=400]
   ```

## Shortcode Syntax

### Basic Embed
```
[html:filename.html]
```

### With Custom Size
```
[html:filename.html width=800 height=600]
```

### With Custom CSS Class
```
[html:filename.html class=centered-embed]
```

### Multiple Parameters
```
[html:demos/calculator.html width=600 height=400 class=bordered-embed]
```

## Supported Parameters

- **width**: Width of iframe (default: 100%)
  - Can be pixels: `width=800` or `width=800px`
  - Can be percentage: `width=80%`

- **height**: Height of iframe (default: 600px)
  - Can be pixels: `height=400` or `height=400px`

- **class**: Custom CSS class for styling the wrapper
  - Available classes: `centered-embed`, `full-width-embed`, `bordered-embed`
  - You can define your own classes in your theme CSS

- **sandbox**: Security level (default: default)
  - `default`: Allows scripts, same-origin access, and forms
  - `strict`: Only allows scripts (more secure, no same-origin access)
  - `permissive`: Allows scripts, same-origin, forms, modals, and popups

## Security

### IFrame Sandboxing
All HTML files are rendered in sandboxed iframes for security. The default sandbox configuration is:

```html
sandbox="allow-scripts allow-same-origin allow-forms"
```

This means:
- ✅ JavaScript execution is enabled
- ✅ Access to cookies and localStorage (same domain)
- ✅ Forms can be submitted
- ❌ Navigation of top-level window is blocked
- ❌ Plugins are blocked

### Security Implications

⚠️ **Important**: With `allow-same-origin` enabled (default mode), embedded HTML files can:
- Access cookies for your domain
- Use localStorage and sessionStorage
- Make fetch/AJAX requests to the same origin
- Communicate with the parent page via postMessage

**Recommendation**: Only embed HTML files from trusted sources!

### File Restrictions

For security, the system:
- Only allows `.html` and `.htm` file extensions
- Blocks directory traversal attempts (e.g., `../../../`)
- Blocks absolute paths and protocol handlers (e.g., `file://`, `http://`)
- Validates all file paths before rendering

## Directory Structure

You can organize your HTML files in subdirectories:

```
/embeds/
├── AtomDiagramTool_Speed.html
├── demos/
│   ├── chart.html
│   └── calculator.html
├── interactive/
│   └── game.html
└── visualizations/
    └── 3d-model.html
```

Then reference them in shortcodes:
```
[html:demos/chart.html]
[html:interactive/game.html]
```

## HTML File Requirements

### Standalone Files
HTML files should be **self-contained** with all resources inline:

```html
<!DOCTYPE html>
<html>
<head>
    <title>My Embed</title>
    <style>
        /* Inline CSS here */
    </style>
</head>
<body>
    <h1>Hello World</h1>

    <script>
        // Inline JavaScript here
    </script>
</body>
</html>
```

### External Resources
If your HTML file needs external resources (images, CSS, JS), you have two options:

1. **Absolute URLs**: Reference external resources with full URLs
   ```html
   <img src="https://example.com/image.png">
   <script src="https://cdn.example.com/library.js"></script>
   ```

2. **Relative Paths**: Place resources in the `/embeds/` directory
   ```
   /embeds/
   ├── mypage.html
   └── assets/
       ├── style.css
       ├── script.js
       └── image.png
   ```

   Then reference them relatively:
   ```html
   <link rel="stylesheet" href="assets/style.css">
   <script src="assets/script.js"></script>
   <img src="assets/image.png">
   ```

## Examples

### Example 1: Simple Embed
```
[html:AtomDiagramTool_Speed.html]
```

### Example 2: Custom Size
```
[html:demos/calculator.html width=600 height=400]
```

### Example 3: Multiple Embeds on Same Page
```markdown
Here's the atom diagram visualization:

[html:AtomDiagramTool_Speed.html]

And here's an interactive calculator:

[html:demos/calculator.html width=500 height=350]
```

### Example 4: With Custom Styling
```
[html:chart.html class=centered-embed]
```

### Example 5: Strict Security Mode
```
[html:untrusted.html sandbox=strict]
```

## Troubleshooting

### "Error: HTML file not found"
- Check that the file exists in the `/embeds/` directory
- Verify the filename and path are correct (case-sensitive on Linux)
- Make sure the file has `.html` or `.htm` extension

### "Error: Invalid file path"
- File path contains invalid characters or directory traversal attempts
- File extension is not `.html` or `.htm`
- File path starts with protocol handler (http://, file://, etc.)

### IFrame appears but content doesn't load
- Check browser console for errors
- Verify the HTML file is valid
- Check for mixed content issues (HTTPS page loading HTTP resources)
- Verify sandbox permissions are sufficient

### JavaScript not working
- Ensure `sandbox` parameter includes `allow-scripts` (default mode does)
- Check browser console for errors
- Verify JavaScript is inline or uses absolute URLs for external scripts

### Can't access parent page from embed
- This is expected with `sandbox=strict` mode
- Use `sandbox=default` (or don't specify sandbox) to enable same-origin access
- Use `postMessage` API for secure parent-child communication

## Communication Between Embeds and Parent Page

If you need your embedded HTML to communicate with the parent page, use the `postMessage` API:

### In the embedded HTML (inside iframe):
```javascript
// Send message to parent
window.parent.postMessage({
    type: 'myEvent',
    data: 'Hello from embed!'
}, '*');
```

### In the parent page:
```javascript
// Listen for messages from embeds
window.addEventListener('message', function(event) {
    if (event.data.type === 'myEvent') {
        console.log(event.data.data);
    }
});
```

## Performance Tips

- HTML embeds use lazy loading by default (iframes below the fold won't load until scrolled into view)
- Keep HTML files small and optimized for faster loading
- Use compression for large inline resources
- Consider loading heavy resources on-demand with JavaScript

## Best Practices

1. **Keep files self-contained**: Inline all CSS and JS when possible
2. **Use semantic HTML**: Proper structure helps with accessibility
3. **Test responsiveness**: Ensure embeds work on mobile devices
4. **Minimize file size**: Smaller files load faster
5. **Use appropriate sandbox mode**: Balance security and functionality
6. **Only embed trusted content**: Avoid embedding untrusted HTML files

## Advanced Usage

### Auto-Resize IFrame
To automatically resize the iframe based on content height, add this to your embedded HTML:

```javascript
// Send height to parent
function updateHeight() {
    const height = document.body.scrollHeight;
    window.parent.postMessage({
        type: 'resize',
        height: height
    }, '*');
}

// Update on load and resize
window.addEventListener('load', updateHeight);
window.addEventListener('resize', updateHeight);
```

### Custom Sandbox Permissions
Need more control? Combine permissions:

```
sandbox=default          → allow-scripts allow-same-origin allow-forms
sandbox=strict           → allow-scripts
sandbox=permissive       → allow-scripts allow-same-origin allow-forms allow-modals allow-popups
```

## Support

For issues or questions about HTML embeds, please contact your BookGram administrator.
