<?php
/**
 * BookGram Plugin System
 *
 * Handles plugin discovery, registration, instance management,
 * shortcode parsing, and rendering for BookGram v10.00
 */

// Storage paths
define('PLUGIN_INSTANCES_DIR', __DIR__ . '/../storage/plugin_instances/');
define('PLUGIN_REGISTRY_FILE', __DIR__ . '/../storage/plugin_registry.json');
define('PLUGIN_AUTH_TOKENS_FILE', __DIR__ . '/../storage/plugin_auth_tokens.json');

// =======================
// PLUGIN DISCOVERY & REGISTRY
// =======================

/**
 * Discover all plugins in the root directory with "plugin_" prefix
 * @return array Associative array of plugins with their manifests
 */
function discover_plugins() {
    $plugins = [];
    $root_dir = dirname(__DIR__, 2); // Go up to bookgram root

    if (!is_dir($root_dir)) {
        return $plugins;
    }

    $items = scandir($root_dir);
    foreach ($items as $item) {
        if (strpos($item, 'plugin_') === 0 && is_dir($root_dir . '/' . $item)) {
            $manifest_path = $root_dir . '/' . $item . '/plugin.json';

            // Check if manifest exists
            if (file_exists($manifest_path)) {
                $manifest_content = file_get_contents($manifest_path);
                $manifest = json_decode($manifest_content, true);

                if ($manifest && json_last_error() === JSON_ERROR_NONE) {
                    $plugins[$item] = [
                        'id' => $item,
                        'path' => $root_dir . '/' . $item,
                        'manifest' => $manifest
                    ];
                }
            }
        }
    }

    return $plugins;
}

/**
 * Load plugin registry from cache
 * @return array Plugin registry data
 */
function load_plugin_registry() {
    if (!file_exists(PLUGIN_REGISTRY_FILE)) {
        return [];
    }

    $content = file_get_contents(PLUGIN_REGISTRY_FILE);
    $registry = json_decode($content, true);

    return ($registry && json_last_error() === JSON_ERROR_NONE) ? $registry : [];
}

/**
 * Save plugin registry to cache
 * @param array $registry Registry data to save
 * @return bool Success status
 */
function save_plugin_registry($registry) {
    $json = json_encode($registry, JSON_PRETTY_PRINT);
    return file_put_contents(PLUGIN_REGISTRY_FILE, $json) !== false;
}

/**
 * Refresh plugin registry by re-discovering plugins
 * @return array Updated registry
 */
function refresh_plugin_registry() {
    $plugins = discover_plugins();
    $registry = [];

    foreach ($plugins as $plugin_id => $plugin_data) {
        $manifest = $plugin_data['manifest'];
        $registry[$plugin_id] = [
            'id' => $plugin_id,
            'name' => $manifest['name'] ?? $plugin_id,
            'enabled' => true,
            'last_updated' => date('c')
        ];
    }

    save_plugin_registry($registry);
    return $registry;
}

// =======================
// INSTANCE PATH MANAGEMENT
// =======================

/**
 * Get the filesystem path for a plugin instance
 * @param string $page_uuid Page UUID
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @return string Full path to instance directory
 */
function get_instance_path($page_uuid, $plugin_id, $instance_id) {
    return PLUGIN_INSTANCES_DIR . $page_uuid . '/' . $plugin_id . '/' . $instance_id;
}

/**
 * Check if a plugin instance exists
 * @param string $page_uuid Page UUID
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @return bool True if instance exists
 */
function instance_exists($page_uuid, $plugin_id, $instance_id) {
    $path = get_instance_path($page_uuid, $plugin_id, $instance_id);
    return file_exists($path . '/instance.json');
}

/**
 * Load instance metadata
 * @param string $page_uuid Page UUID
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @return array|null Instance metadata or null if not found
 */
function load_instance_metadata($page_uuid, $plugin_id, $instance_id) {
    if (!instance_exists($page_uuid, $plugin_id, $instance_id)) {
        return null;
    }

    $path = get_instance_path($page_uuid, $plugin_id, $instance_id);
    $content = file_get_contents($path . '/instance.json');
    $metadata = json_decode($content, true);

    return ($metadata && json_last_error() === JSON_ERROR_NONE) ? $metadata : null;
}

/**
 * Save instance metadata
 * @param string $page_uuid Page UUID
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @param array $metadata Metadata to save
 * @return bool Success status
 */
function save_instance_metadata($page_uuid, $plugin_id, $instance_id, $metadata) {
    $path = get_instance_path($page_uuid, $plugin_id, $instance_id);
    $json = json_encode($metadata, JSON_PRETTY_PRINT);
    return file_put_contents($path . '/instance.json', $json) !== false;
}

// =======================
// INSTANCE LIFECYCLE
// =======================

/**
 * Create a new plugin instance
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @param string $page_uuid Page UUID
 * @param array $params Optional parameters
 * @return string|false Instance path on success, false on failure
 */
function create_plugin_instance($plugin_id, $instance_id, $page_uuid, $params = []) {
    $instance_path = get_instance_path($page_uuid, $plugin_id, $instance_id);

    // Create directory structure
    if (!is_dir($instance_path)) {
        if (!mkdir($instance_path, 0755, true)) {
            return false;
        }
    }

    // Create data subdirectory
    $data_dir = $instance_path . '/data';
    if (!is_dir($data_dir)) {
        if (!mkdir($data_dir, 0755, true)) {
            return false;
        }
    }

    // Get current username (from auth session)
    $username = 'system';
    if (isset($_SESSION['user'])) {
        $username = $_SESSION['user'];
    }

    // Create metadata
    $metadata = [
        'instance_id' => $instance_id,
        'plugin_id' => $plugin_id,
        'page_uuid' => $page_uuid,
        'created_at' => date('c'),
        'created_by' => $username,
        'parameters' => $params,
        'initialized' => false,
        'is_private' => false,
        'share_token' => '',
        'password_hash' => ''
    ];

    $json = json_encode($metadata, JSON_PRETTY_PRINT);
    if (file_put_contents($instance_path . '/instance.json', $json) === false) {
        return false;
    }

    return $instance_path;
}

/**
 * Delete a plugin instance and all its data
 * @param string $page_uuid Page UUID
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @return bool Success status
 */
function delete_plugin_instance($page_uuid, $plugin_id, $instance_id) {
    $path = get_instance_path($page_uuid, $plugin_id, $instance_id);

    if (!is_dir($path)) {
        return false;
    }

    // Use BookGram's existing recursive delete function
    require_once __DIR__ . '/functions.php';

    // Recursive delete helper (inline since we can't guarantee function exists)
    $delete_recursive = function($dir) use (&$delete_recursive) {
        if (!is_dir($dir)) {
            return unlink($dir);
        }

        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $delete_recursive($path) : unlink($path);
        }

        return rmdir($dir);
    };

    return $delete_recursive($path);
}

/**
 * Initialize a plugin instance by calling its setup.php
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @param string $page_uuid Page UUID
 * @param array $params Parameters to pass to setup
 * @return bool Success status
 */
function initialize_plugin_instance($plugin_id, $instance_id, $page_uuid, $params = []) {
    $instance_path = get_instance_path($page_uuid, $plugin_id, $instance_id);

    // Create instance if it doesn't exist
    if (!instance_exists($page_uuid, $plugin_id, $instance_id)) {
        if (create_plugin_instance($plugin_id, $instance_id, $page_uuid, $params) === false) {
            return false;
        }
    }

    // Load metadata
    $metadata = load_instance_metadata($page_uuid, $plugin_id, $instance_id);
    if (!$metadata) {
        return false;
    }

    // Skip if already initialized
    if ($metadata['initialized']) {
        return true;
    }

    // Get plugin info
    $plugins = discover_plugins();
    if (!isset($plugins[$plugin_id])) {
        return false;
    }

    $plugin = $plugins[$plugin_id];
    $setup_script = $plugin['path'] . '/' . ($plugin['manifest']['setup_script'] ?? 'setup.php');

    // Check if setup script exists
    if (!file_exists($setup_script)) {
        // No setup needed, mark as initialized
        $metadata['initialized'] = true;
        save_instance_metadata($page_uuid, $plugin_id, $instance_id, $metadata);
        return true;
    }

    // Set environment variables for plugin context
    $_ENV['BOOKGRAM_INSTANCE_PATH'] = $instance_path;
    $_ENV['BOOKGRAM_INSTANCE_ID'] = $instance_id;
    $_ENV['BOOKGRAM_PAGE_UUID'] = $page_uuid;
    $_ENV['BOOKGRAM_PARAMS'] = json_encode($params);

    // Execute setup script
    try {
        ob_start();
        include $setup_script;
        ob_end_clean();

        // Mark as initialized
        $metadata['initialized'] = true;
        save_instance_metadata($page_uuid, $plugin_id, $instance_id, $metadata);

        return true;
    } catch (Exception $e) {
        ob_end_clean();
        error_log('Plugin setup failed: ' . $e->getMessage());
        return false;
    }
}

// =======================
// SHORTCODE PARSING
// =======================

/**
 * Extract shortcodes from content
 * @param string $content Article content
 * @return array Array of shortcode instances found
 */
function extract_shortcodes_from_content($content) {
    $instances = [];
    $pattern = '/\[plugin:([a-z0-9_]+)\s+id=([a-z0-9_-]+)([^\]]*)\]/i';

    preg_match_all($pattern, $content, $matches, PREG_SET_ORDER);

    foreach ($matches as $match) {
        $plugin_name = $match[1];
        $instance_id = $match[2];
        $params_string = $match[3] ?? '';

        $instances[] = [
            'plugin' => 'plugin_' . $plugin_name,
            'id' => $instance_id,
            'params' => parse_shortcode_params($params_string)
        ];
    }

    return $instances;
}

/**
 * Parse shortcode parameters from string
 * @param string $params_string Parameter string (e.g., "key1=value1 key2=value2")
 * @return array Associative array of parameters
 */
function parse_shortcode_params($params_string) {
    $params = [];
    $params_string = trim($params_string);

    if (empty($params_string)) {
        return $params;
    }

    preg_match_all('/([a-z0-9_-]+)=([a-z0-9_-]+)/i', $params_string, $matches, PREG_SET_ORDER);

    foreach ($matches as $match) {
        $params[$match[1]] = $match[2];
    }

    return $params;
}

/**
 * Parse shortcodes in content and replace with rendered plugin output
 * @param string $content Article content with shortcodes
 * @param string $page_uuid Page UUID
 * @return string Content with shortcodes replaced by plugin output
 */
function parse_shortcodes($content, $page_uuid) {
    $pattern = '/\[plugin:([a-z0-9_]+)\s+id=([a-z0-9_-]+)([^\]]*)\]/i';

    return preg_replace_callback($pattern, function($matches) use ($page_uuid) {
        $plugin_name = $matches[1];
        $instance_id = $matches[2];
        $params_string = $matches[3] ?? '';

        // Build full plugin ID
        $plugin_id = 'plugin_' . $plugin_name;

        // Parse parameters
        $params = parse_shortcode_params($params_string);
        $params['id'] = $instance_id; // Ensure ID is in params

        // Render plugin instance
        return render_plugin_instance($plugin_id, $instance_id, $page_uuid, $params);
    }, $content);
}

// =======================
// HTML EMBED SYSTEM
// =======================

/**
 * Extract HTML embed shortcodes from content
 * @param string $content Article content
 * @return array Array of HTML embed instances found
 */
function extract_html_embeds_from_content($content) {
    $embeds = [];
    $pattern = '/\[html:([^\]]+)\]/i';

    preg_match_all($pattern, $content, $matches, PREG_SET_ORDER);

    foreach ($matches as $match) {
        $params_string = $match[1];
        $parts = preg_split('/\s+/', $params_string, 2);
        $filepath = $parts[0];
        $params_str = isset($parts[1]) ? $parts[1] : '';

        $embeds[] = [
            'filepath' => $filepath,
            'params' => parse_html_embed_params($params_str),
            'full_match' => $match[0]
        ];
    }

    return $embeds;
}

/**
 * Parse HTML embed parameters from string
 * @param string $params_string Parameter string (e.g., "width=800 height=600")
 * @return array Associative array of parameters
 */
function parse_html_embed_params($params_string) {
    $params = [];
    $params_string = trim($params_string);

    if (empty($params_string)) {
        return $params;
    }

    preg_match_all('/([a-z0-9_-]+)=([a-z0-9_%-]+)/i', $params_string, $matches, PREG_SET_ORDER);

    foreach ($matches as $match) {
        $params[$match[1]] = $match[2];
    }

    return $params;
}

/**
 * Sanitize HTML embed filepath
 * @param string $filepath User-provided filepath
 * @return string|false Sanitized path or false if invalid
 */
function sanitize_html_embed_path($filepath) {
    // Remove any directory traversal attempts
    $filepath = str_replace('..', '', $filepath);
    $filepath = str_replace('//', '/', $filepath);
    $filepath = trim($filepath, '/');

    // Validate extension
    $ext = strtolower(pathinfo($filepath, PATHINFO_EXTENSION));
    if (!in_array($ext, ['html', 'htm'])) {
        return false;
    }

    // Ensure no absolute paths or protocol handlers
    if (preg_match('/^([a-z]:|\\/|https?:|file:|data:)/i', $filepath)) {
        return false;
    }

    return $filepath;
}

/**
 * Check if HTML embed file exists
 * @param string $filepath Sanitized filepath
 * @return bool True if file exists
 */
function html_embed_exists($filepath) {
    $embeds_dir = dirname(__DIR__, 2) . '/embeds/';
    $full_path = $embeds_dir . $filepath;

    return file_exists($full_path) && is_file($full_path);
}

/**
 * Render an HTML embed as iframe
 * @param string $filepath Filepath relative to embeds directory
 * @param array $params Parameters from shortcode
 * @return string Rendered HTML iframe
 */
function render_html_embed($filepath, $params = []) {
    // Sanitize filepath
    $safe_filepath = sanitize_html_embed_path($filepath);

    if ($safe_filepath === false) {
        return '<div class="html-embed-error">Error: Invalid file path</div>';
    }

    // Check if file exists
    if (!html_embed_exists($safe_filepath)) {
        return '<div class="html-embed-error">Error: HTML file not found: ' .
               htmlspecialchars($filepath, ENT_QUOTES) . '</div>';
    }

    // Build iframe attributes
    $width = isset($params['width']) ? $params['width'] : '100%';
    $height = isset($params['height']) ? $params['height'] : '600px';
    $custom_class = isset($params['class']) ? $params['class'] : '';
    $sandbox_mode = isset($params['sandbox']) ? $params['sandbox'] : 'default';

    // Normalize dimensions
    if (is_numeric($width)) {
        $width .= 'px';
    }
    if (is_numeric($height)) {
        $height .= 'px';
    }

    // Sandbox attribute
    $sandbox_attrs = [
        'default' => 'allow-scripts allow-same-origin allow-forms',
        'strict' => 'allow-scripts',
        'permissive' => 'allow-scripts allow-same-origin allow-forms allow-modals allow-popups'
    ];
    $sandbox = isset($sandbox_attrs[$sandbox_mode]) ? $sandbox_attrs[$sandbox_mode] : $sandbox_attrs['default'];

    // Build iframe URL
    $iframe_src = '/embeds/' . htmlspecialchars($safe_filepath, ENT_QUOTES);

    // Generate unique ID for this embed
    $embed_id = 'html-embed-' . md5($safe_filepath . microtime());

    // Build wrapper classes
    $wrapper_classes = 'bookgram-html-embed-wrapper';
    if (!empty($custom_class)) {
        $wrapper_classes .= ' ' . htmlspecialchars($custom_class, ENT_QUOTES);
    }

    // Render iframe
    $html = '<div class="' . $wrapper_classes . '" data-embed-file="' . htmlspecialchars($safe_filepath, ENT_QUOTES) . '">' . "\n";
    $html .= '    <iframe' . "\n";
    $html .= '        id="' . $embed_id . '"' . "\n";
    $html .= '        src="' . $iframe_src . '"' . "\n";
    $html .= '        width="' . $width . '"' . "\n";
    $html .= '        height="' . $height . '"' . "\n";
    $html .= '        sandbox="' . $sandbox . '"' . "\n";
    $html .= '        frameborder="0"' . "\n";
    $html .= '        loading="lazy"' . "\n";
    $html .= '        class="html-embed-iframe"' . "\n";
    $html .= '        title="Embedded HTML: ' . htmlspecialchars($safe_filepath, ENT_QUOTES) . '">' . "\n";
    $html .= '        Your browser does not support iframes.' . "\n";
    $html .= '    </iframe>' . "\n";
    $html .= '</div>';

    return $html;
}

/**
 * Parse HTML embed shortcodes in content
 * @param string $content Article content with HTML embed shortcodes
 * @return string Content with shortcodes replaced by iframes
 */
function parse_html_embeds($content) {
    $pattern = '/\[html:([^\]]+)\]/i';

    return preg_replace_callback($pattern, function($matches) {
        $params_string = $matches[1];
        $parts = preg_split('/\s+/', $params_string, 2);
        $filepath = $parts[0];
        $params_str = isset($parts[1]) ? $parts[1] : '';

        // Parse parameters
        $params = parse_html_embed_params($params_str);

        // Render embed
        return render_html_embed($filepath, $params);
    }, $content);
}

// =======================
// PHP EMBED SYSTEM
// =======================

/**
 * Extract PHP embed shortcodes from content
 * @param string $content Article content
 * @return array Array of PHP embed instances found
 */
function extract_php_embeds_from_content($content) {
    $embeds = [];
    $pattern = '/\[php:([^\]]+)\]/i';

    preg_match_all($pattern, $content, $matches, PREG_SET_ORDER);

    foreach ($matches as $match) {
        $params_string = $match[1];
        $parts = preg_split('/\s+/', $params_string, 2);
        $filepath = $parts[0];
        $params_str = isset($parts[1]) ? $parts[1] : '';

        $embeds[] = [
            'filepath' => $filepath,
            'params' => parse_php_embed_params($params_str),
            'full_match' => $match[0]
        ];
    }

    return $embeds;
}

/**
 * Parse PHP embed parameters from string
 * @param string $params_string Parameter string (e.g., "micro=WISDOM meso=LOVE width=800")
 * @return array Associative array of parameters
 */
function parse_php_embed_params($params_string) {
    $params = [];
    $params_string = trim($params_string);

    if (empty($params_string)) {
        return $params;
    }

    preg_match_all('/([a-z0-9_-]+)=([a-z0-9_%-]+)/i', $params_string, $matches, PREG_SET_ORDER);

    foreach ($matches as $match) {
        $params[$match[1]] = $match[2];
    }

    return $params;
}

/**
 * Sanitize PHP embed filepath
 * @param string $filepath User-provided filepath
 * @return string|false Sanitized path or false if invalid
 */
function sanitize_php_embed_path($filepath) {
    // Remove any directory traversal attempts
    $filepath = str_replace('..', '', $filepath);
    $filepath = str_replace('//', '/', $filepath);
    $filepath = trim($filepath, '/');

    // Validate extension
    $ext = strtolower(pathinfo($filepath, PATHINFO_EXTENSION));
    if ($ext !== 'php') {
        return false;
    }

    // Ensure no absolute paths or protocol handlers
    if (preg_match('/^([a-z]:|\\/|https?:|file:|data:|php:)/i', $filepath)) {
        return false;
    }

    return $filepath;
}

/**
 * Check if PHP embed file exists
 * @param string $filepath Sanitized filepath
 * @return bool True if file exists
 */
function php_embed_exists($filepath) {
    $php_embeds_dir = dirname(__DIR__, 2) . '/php_embeds/';
    $full_path = $php_embeds_dir . $filepath;

    return file_exists($full_path) && is_file($full_path);
}

/**
 * Get PHP embed metadata from file comments
 * @param string $filepath Filepath relative to php_embeds directory
 * @return array Metadata including parameter definitions
 */
function get_php_embed_metadata($filepath) {
    $php_embeds_dir = dirname(__DIR__, 2) . '/php_embeds/';
    $full_path = $php_embeds_dir . $filepath;

    if (!file_exists($full_path)) {
        return ['parameters' => []];
    }

    $content = file_get_contents($full_path);
    $metadata = ['parameters' => []];

    // Parse BOOKGRAM_PARAM annotations
    // Format: BOOKGRAM_PARAM: name | type | options | default
    preg_match_all('/BOOKGRAM_PARAM:\s*([a-z0-9_-]+)\s*\|\s*([a-z]+)\s*\|\s*([^\|]*)\s*\|\s*([^\n]*)/i',
                    $content, $matches, PREG_SET_ORDER);

    foreach ($matches as $match) {
        $param_name = trim($match[1]);
        $param_type = trim($match[2]);
        $param_options = trim($match[3]);
        $param_default = trim($match[4]);

        $param_config = [
            'type' => $param_type,
            'default' => $param_default,
            'label' => ucfirst(str_replace('_', ' ', $param_name))
        ];

        // Parse options based on type
        if ($param_type === 'select' && !empty($param_options)) {
            $param_config['options'] = array_map('trim', explode(',', $param_options));
        } elseif ($param_type === 'range' && !empty($param_options)) {
            // Format: "0.1-2.0" or "0-100"
            if (preg_match('/^([0-9.]+)-([0-9.]+)$/', $param_options, $range_match)) {
                $param_config['min'] = floatval($range_match[1]);
                $param_config['max'] = floatval($range_match[2]);
                // Auto-detect step based on decimal places
                $has_decimal = (strpos($param_options, '.') !== false);
                $param_config['step'] = $has_decimal ? 0.1 : 1;
            }
        }

        $metadata['parameters'][$param_name] = $param_config;
    }

    return $metadata;
}

/**
 * Get comprehensive list of blacklisted PHP functions
 * @return array List of forbidden function names
 */
function get_php_function_blacklist() {
    return [
        // File operations
        'fopen', 'fwrite', 'fclose', 'file_get_contents', 'file_put_contents',
        'file', 'readfile', 'fread', 'fgets', 'fgetc', 'fgetcsv',
        'fputs', 'fputcsv', 'unlink', 'rmdir', 'mkdir', 'rename',
        'copy', 'move_uploaded_file', 'tmpfile', 'tempnam',

        // Directory operations
        'chdir', 'chroot', 'dir', 'opendir', 'readdir', 'closedir',
        'scandir', 'glob',

        // Shell/system
        'exec', 'system', 'passthru', 'shell_exec', 'popen', 'proc_open',
        'proc_close', 'proc_terminate', 'proc_get_status', 'pcntl_exec',

        // Code evaluation
        'eval', 'assert', 'create_function', 'include', 'include_once',
        'require', 'require_once',

        // Network
        'fsockopen', 'pfsockopen', 'socket_create', 'socket_connect',
        'curl_exec', 'curl_multi_exec', 'curl_init',

        // Database
        'mysqli_connect', 'mysqli_query', 'mysqli_real_connect',
        'mysql_connect', 'mysql_query', 'pg_connect', 'pg_query',
        'sqlite_open', 'PDO',

        // Dangerous PHP functions
        'dl', 'extract', 'parse_ini_file', 'show_source', 'highlight_file',
        'php_uname', 'getenv', 'putenv', 'ini_set', 'ini_alter', 'ini_restore',
        'set_time_limit', 'ignore_user_abort',

        // Serialization
        'unserialize', 'serialize',

        // Session/cookies
        'session_start', 'session_destroy', 'setcookie', 'setrawcookie',

        // Output control
        'header', 'header_remove', 'headers_sent', 'http_response_code',

        // Reflection (can bypass restrictions)
        'ReflectionFunction', 'ReflectionClass', 'ReflectionMethod'
    ];
}

/**
 * Scan PHP code for blacklisted functions
 * @param string $filepath Filepath relative to php_embeds directory
 * @return array Array of violations (function name and line number)
 */
function scan_php_code_for_blacklisted_functions($filepath) {
    $php_embeds_dir = dirname(__DIR__, 2) . '/php_embeds/';
    $full_path = $php_embeds_dir . $filepath;

    if (!file_exists($full_path)) {
        return [['function' => 'file_not_found', 'line' => 0]];
    }

    $code = file_get_contents($full_path);
    $blacklist = get_php_function_blacklist();
    $blacklist_lower = array_map('strtolower', $blacklist);
    $violations = [];

    // Tokenize PHP code
    $tokens = token_get_all($code);

    // Language constructs that need special detection
    $language_constructs = [
        T_EVAL => 'eval',
        T_INCLUDE => 'include',
        T_INCLUDE_ONCE => 'include_once',
        T_REQUIRE => 'require',
        T_REQUIRE_ONCE => 'require_once'
    ];

    for ($i = 0; $i < count($tokens); $i++) {
        if (!is_array($tokens[$i])) {
            continue;
        }

        $token_type = $tokens[$i][0];
        $line_number = $tokens[$i][2];

        // Check for language constructs
        if (isset($language_constructs[$token_type])) {
            $construct_name = $language_constructs[$token_type];
            if (in_array($construct_name, $blacklist_lower)) {
                $violations[] = [
                    'function' => $construct_name,
                    'line' => $line_number
                ];
            }
            continue;
        }

        // Check for regular function calls
        if ($token_type === T_STRING) {
            $function_name = $tokens[$i][1];

            // Check if next non-whitespace token is '('
            $j = $i + 1;
            while ($j < count($tokens) && is_array($tokens[$j]) && $tokens[$j][0] === T_WHITESPACE) {
                $j++;
            }

            if ($j < count($tokens) && $tokens[$j] === '(') {
                // This is a function call
                if (in_array(strtolower($function_name), $blacklist_lower)) {
                    $violations[] = [
                        'function' => $function_name,
                        'line' => $line_number
                    ];
                }
            }
        }
    }

    return $violations;
}

/**
 * Generate HTML form controls from parameter metadata
 * @param array $metadata Parameter metadata from get_php_embed_metadata()
 * @param string $embed_id Unique embed ID
 * @param array $current_values Current parameter values
 * @return string HTML form controls
 */
function generate_php_param_controls($metadata, $embed_id, $current_values = []) {
    $html = '';

    if (empty($metadata['parameters'])) {
        return $html;
    }

    $html .= '<div class="php-embed-controls" id="controls-' . $embed_id . '">' . "\n";
    $html .= '<form class="php-param-form" onsubmit="return updatePhpEmbed(\'' . $embed_id . '\', event);">' . "\n";

    foreach ($metadata['parameters'] as $param_name => $param_config) {
        $current_value = $current_values[$param_name] ?? $param_config['default'];
        $label = $param_config['label'] ?? ucfirst(str_replace('_', ' ', $param_name));

        $html .= '<div class="param-control">' . "\n";
        $html .= '<label for="param-' . $embed_id . '-' . htmlspecialchars($param_name, ENT_QUOTES) . '">' .
                 htmlspecialchars($label, ENT_QUOTES) . ':</label>' . "\n";

        switch ($param_config['type']) {
            case 'select':
                $html .= '<select id="param-' . $embed_id . '-' . htmlspecialchars($param_name, ENT_QUOTES) .
                         '" name="' . htmlspecialchars($param_name, ENT_QUOTES) . '">' . "\n";
                foreach ($param_config['options'] as $option) {
                    $selected = ($option == $current_value) ? ' selected' : '';
                    $html .= '<option value="' . htmlspecialchars($option, ENT_QUOTES) . '"' . $selected . '>' .
                             htmlspecialchars($option, ENT_QUOTES) . '</option>' . "\n";
                }
                $html .= '</select>' . "\n";
                break;

            case 'range':
                $html .= '<input type="range" ';
                $html .= 'id="param-' . $embed_id . '-' . htmlspecialchars($param_name, ENT_QUOTES) . '" ';
                $html .= 'name="' . htmlspecialchars($param_name, ENT_QUOTES) . '" ';
                $html .= 'min="' . $param_config['min'] . '" ';
                $html .= 'max="' . $param_config['max'] . '" ';
                $html .= 'step="' . ($param_config['step'] ?? 0.1) . '" ';
                $html .= 'value="' . htmlspecialchars($current_value, ENT_QUOTES) . '" ';
                $html .= 'oninput="document.getElementById(\'value-' . $embed_id . '-' .
                         htmlspecialchars($param_name, ENT_QUOTES) . '\').textContent = this.value">';
                $html .= '<span id="value-' . $embed_id . '-' . htmlspecialchars($param_name, ENT_QUOTES) . '">' .
                         htmlspecialchars($current_value, ENT_QUOTES) . '</span>' . "\n";
                break;

            case 'text':
                $html .= '<input type="text" ';
                $html .= 'id="param-' . $embed_id . '-' . htmlspecialchars($param_name, ENT_QUOTES) . '" ';
                $html .= 'name="' . htmlspecialchars($param_name, ENT_QUOTES) . '" ';
                $html .= 'value="' . htmlspecialchars($current_value, ENT_QUOTES) . '">' . "\n";
                break;
        }

        $html .= '</div>' . "\n";
    }

    $html .= '<button type="submit" class="update-btn">Update</button>' . "\n";
    $html .= '</form>' . "\n";
    $html .= '</div>' . "\n";

    return $html;
}

/**
 * Render a PHP embed as iframe with controls
 * @param string $filepath Filepath relative to php_embeds directory
 * @param array $params Parameters from shortcode
 * @return string Rendered HTML iframe with controls
 */
function render_php_embed($filepath, $params = []) {
    // Sanitize filepath
    $safe_filepath = sanitize_php_embed_path($filepath);

    if ($safe_filepath === false) {
        return '<div class="php-embed-error">Error: Invalid file path</div>';
    }

    // Check if file exists
    if (!php_embed_exists($safe_filepath)) {
        return '<div class="php-embed-error">Error: PHP file not found: ' .
               htmlspecialchars($filepath, ENT_QUOTES) . '</div>';
    }

    // Scan for blacklisted functions
    $violations = scan_php_code_for_blacklisted_functions($safe_filepath);
    if (!empty($violations)) {
        $violation_list = implode(', ', array_column($violations, 'function'));
        return '<div class="php-embed-error">Security Error: File contains forbidden functions: ' .
               htmlspecialchars($violation_list, ENT_QUOTES) . '</div>';
    }

    // Get metadata for parameter controls
    $metadata = get_php_embed_metadata($safe_filepath);

    // Build iframe attributes
    $width = isset($params['width']) ? $params['width'] : '100%';
    $height = isset($params['height']) ? $params['height'] : '600px';
    $custom_class = isset($params['class']) ? $params['class'] : '';

    // Normalize dimensions
    if (is_numeric($width)) {
        $width .= 'px';
    }
    if (is_numeric($height)) {
        $height .= 'px';
    }

    // Generate unique ID
    $embed_id = 'php-embed-' . md5($safe_filepath . microtime());

    // Build execution URL with parameters
    $exec_url = '/php-execute?file=' . urlencode($safe_filepath);
    foreach ($params as $key => $value) {
        if (!in_array($key, ['width', 'height', 'class'])) {
            $exec_url .= '&' . urlencode($key) . '=' . urlencode($value);
        }
    }

    // Sandbox attribute (strict for PHP execution)
    $sandbox = 'allow-scripts';

    // Build wrapper classes
    $wrapper_classes = 'bookgram-php-embed-wrapper';
    if (!empty($custom_class)) {
        $wrapper_classes .= ' ' . htmlspecialchars($custom_class, ENT_QUOTES);
    }

    // Generate parameter controls if metadata exists
    $controls_html = generate_php_param_controls($metadata, $embed_id, $params);

    // Render iframe with controls
    $html = '<div class="' . $wrapper_classes . '" data-embed-file="' .
            htmlspecialchars($safe_filepath, ENT_QUOTES) . '">' . "\n";

    // Parameter controls above iframe
    if (!empty($controls_html)) {
        $html .= $controls_html;
    }

    // Iframe
    $html .= '    <iframe' . "\n";
    $html .= '        id="' . $embed_id . '"' . "\n";
    $html .= '        src="' . $exec_url . '"' . "\n";
    $html .= '        width="' . $width . '"' . "\n";
    $html .= '        height="' . $height . '"' . "\n";
    $html .= '        sandbox="' . $sandbox . '"' . "\n";
    $html .= '        frameborder="0"' . "\n";
    $html .= '        loading="lazy"' . "\n";
    $html .= '        class="php-embed-iframe"' . "\n";
    $html .= '        title="PHP Embed: ' . htmlspecialchars($safe_filepath, ENT_QUOTES) . '">' . "\n";
    $html .= '        Your browser does not support iframes.' . "\n";
    $html .= '    </iframe>' . "\n";
    $html .= '</div>';

    return $html;
}

/**
 * Parse PHP embed shortcodes in content
 * @param string $content Article content with PHP embed shortcodes
 * @return string Content with shortcodes replaced by iframes
 */
function parse_php_embeds($content) {
    $pattern = '/\[php:([^\]]+)\]/i';

    return preg_replace_callback($pattern, function($matches) {
        $params_string = $matches[1];
        $parts = preg_split('/\s+/', $params_string, 2);
        $filepath = $parts[0];
        $params_str = isset($parts[1]) ? $parts[1] : '';

        // Parse parameters
        $params = parse_php_embed_params($params_str);

        // Render embed
        return render_php_embed($filepath, $params);
    }, $content);
}

// =======================
// PLUGIN RENDERING
// =======================

/**
 * Render a plugin instance
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @param string $page_uuid Page UUID
 * @param array $params Parameters from shortcode
 * @return string Rendered HTML
 */
function render_plugin_instance($plugin_id, $instance_id, $page_uuid, $params = []) {
    // Initialize instance if needed
    if (!initialize_plugin_instance($plugin_id, $instance_id, $page_uuid, $params)) {
        return '<div class="plugin-error">Error: Failed to initialize plugin instance</div>';
    }

    // Load instance metadata
    $metadata = load_instance_metadata($page_uuid, $plugin_id, $instance_id);
    if (!$metadata) {
        return '<div class="plugin-error">Error: Plugin instance not found</div>';
    }

    // Check privacy settings
    if (!can_view_plugin_instance($metadata)) {
        return '<div class="plugin-error">This plugin instance is private. Access denied.</div>';
    }

    // Get plugin info
    $plugins = discover_plugins();
    if (!isset($plugins[$plugin_id])) {
        return '<div class="plugin-error">Plugin not found: ' . htmlspecialchars($plugin_id) . '</div>';
    }

    $plugin = $plugins[$plugin_id];
    $instance_path = get_instance_path($page_uuid, $plugin_id, $instance_id);

    // Check for embed handler
    $embed_handler = $plugin['path'] . '/' . ($plugin['manifest']['embed_handler'] ?? 'embed.php');

    if (!file_exists($embed_handler)) {
        return '<div class="plugin-error">Plugin embed handler not found</div>';
    }

    // Set environment variables for plugin context
    $_ENV['BOOKGRAM_INSTANCE_PATH'] = $instance_path;
    $_ENV['BOOKGRAM_INSTANCE_ID'] = $instance_id;
    $_ENV['BOOKGRAM_PAGE_UUID'] = $page_uuid;
    $_ENV['BOOKGRAM_PARAMS'] = json_encode($params);

    // Capture plugin output
    ob_start();
    try {
        include $embed_handler;
        $plugin_output = ob_get_clean();
    } catch (Exception $e) {
        ob_end_clean();
        $plugin_output = '<div class="plugin-error">Error rendering plugin: ' . htmlspecialchars($e->getMessage()) . '</div>';
    }

    // Wrap with CSS isolation
    return wrap_plugin_output($plugin_output, $plugin_id, $instance_id, $plugin);
}

/**
 * Wrap plugin output with CSS isolation
 * @param string $content Plugin HTML content
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @param array $plugin Plugin data with manifest
 * @return string Wrapped HTML
 */
function wrap_plugin_output($content, $plugin_id, $instance_id, $plugin) {
    $wrapper_class = 'bookgram-plugin-wrapper plugin-' . sanitize_css_class($plugin_id);
    $instance_class = 'instance-' . sanitize_css_class($instance_id);

    // Load plugin CSS files
    $plugin_css = '';
    if (isset($plugin['manifest']['css_files'])) {
        foreach ($plugin['manifest']['css_files'] as $css_file) {
            $css_path = '/' . $plugin_id . '/' . $css_file;
            $plugin_css .= '<link rel="stylesheet" href="' . htmlspecialchars($css_path) . '">' . "\n";
        }
    }

    return <<<HTML
<div class="{$wrapper_class} {$instance_class}" data-plugin="{$plugin_id}" data-instance="{$instance_id}">
    {$plugin_css}
    <div class="plugin-content">
        {$content}
    </div>
</div>
HTML;
}

/**
 * Sanitize string for use as CSS class name
 * @param string $string Input string
 * @return string Sanitized CSS class name
 */
function sanitize_css_class($string) {
    return preg_replace('/[^a-z0-9_-]/i', '-', $string);
}

// =======================
// PRIVACY CONTROLS
// =======================

/**
 * Check if current user can view a plugin instance
 * @param array $metadata Instance metadata
 * @return bool True if user can view
 */
function can_view_plugin_instance($metadata) {
    // Public instances are always viewable
    if (empty($metadata['is_private'])) {
        return true;
    }

    // Check if user is logged in and is admin/editor
    if (isset($_SESSION['user'])) {
        if (function_exists('is_admin') && is_admin()) {
            return true;
        }

        // Check if user is the creator
        if (isset($_SESSION['user']) && $_SESSION['user'] === $metadata['created_by']) {
            return true;
        }
    }

    // Check for valid share token
    if (!empty($metadata['share_token'])) {
        $provided_token = $_GET['token'] ?? '';
        if ($provided_token === $metadata['share_token']) {
            // Check password if required
            if (!empty($metadata['password_hash'])) {
                // Check if password was already verified in session
                $session_key = 'plugin_instance_password_' . $metadata['instance_id'];
                if (isset($_SESSION[$session_key]) && $_SESSION[$session_key] === true) {
                    return true;
                }
                return false; // Password required but not verified
            }
            return true;
        }
    }

    return false;
}

/**
 * Generate a unique share token for a plugin instance
 * @return string Random token
 */
function generate_instance_share_token() {
    return bin2hex(random_bytes(16));
}

/**
 * Verify instance password
 * @param array $metadata Instance metadata
 * @param string $password Password to verify
 * @return bool True if password is correct
 */
function verify_instance_password($metadata, $password) {
    if (empty($metadata['password_hash'])) {
        return true; // No password set
    }

    return password_verify($password, $metadata['password_hash']);
}

// =======================
// AUTH TOKEN SYSTEM
// =======================

/**
 * Create an auth token for plugin admin access
 * @param string $page_uuid Page UUID
 * @param string $plugin_id Plugin ID
 * @param string $instance_id Instance ID
 * @return string|false Token on success, false on failure
 */
function create_plugin_auth_token($page_uuid, $plugin_id, $instance_id) {
    if (!isset($_SESSION['user']) || !isset($_SESSION['role'])) {
        return false;
    }

    $token_data = [
        'username' => $_SESSION['user'],
        'role' => $_SESSION['role'],
        'page_uuid' => $page_uuid,
        'plugin_id' => $plugin_id,
        'instance_id' => $instance_id,
        'expires' => time() + 3600, // 1 hour
        'token' => bin2hex(random_bytes(32))
    ];

    // Load existing tokens
    $tokens = [];
    if (file_exists(PLUGIN_AUTH_TOKENS_FILE)) {
        $content = file_get_contents(PLUGIN_AUTH_TOKENS_FILE);
        $tokens = json_decode($content, true) ?: [];
    }

    // Add new token
    $tokens[$token_data['token']] = $token_data;

    // Clean expired tokens
    foreach ($tokens as $key => $data) {
        if ($data['expires'] < time()) {
            unset($tokens[$key]);
        }
    }

    // Save tokens
    $json = json_encode($tokens, JSON_PRETTY_PRINT);
    if (file_put_contents(PLUGIN_AUTH_TOKENS_FILE, $json) === false) {
        return false;
    }

    return $token_data['token'];
}

/**
 * Verify a plugin auth token
 * @param string $token Token to verify
 * @return array|false Token data on success, false on failure
 */
function verify_plugin_auth_token($token) {
    if (!file_exists(PLUGIN_AUTH_TOKENS_FILE)) {
        return false;
    }

    $content = file_get_contents(PLUGIN_AUTH_TOKENS_FILE);
    $tokens = json_decode($content, true);

    if (!$tokens || !isset($tokens[$token])) {
        return false;
    }

    $data = $tokens[$token];

    // Check expiration
    if ($data['expires'] < time()) {
        return false;
    }

    return $data;
}
