<?php
/**
 * BookGram Plugin Authentication Helper
 *
 * This file should be included in plugin admin pages to authenticate
 * against BookGram's session system via auth tokens.
 *
 * Usage in plugin admin pages:
 *   require_once '../../app/core/plugin_auth.php';
 *   $auth = bookgram_auth_check();
 *   bookgram_require_auth();
 */

/**
 * Check BookGram authentication via token
 * Verifies the bga_token parameter and sets up plugin session
 *
 * @return array|false Auth data on success, false on failure
 */
function bookgram_auth_check() {
    // Start session if not already started
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // Get token from URL parameter
    $token = $_GET['bga_token'] ?? null;

    // If we have a token, always verify it and update session
    // This ensures session data is refreshed on each page load
    if ($token) {
        // Load plugin system functions
        if (!function_exists('verify_plugin_auth_token')) {
            require_once __DIR__ . '/plugins.php';
        }

        // Verify token
        $auth_data = verify_plugin_auth_token($token);

        if ($auth_data) {
            // Set session variables
            $_SESSION['bookgram_user'] = $auth_data['username'];
            $_SESSION['bookgram_role'] = $auth_data['role'];
            $_SESSION['bookgram_instance_id'] = $auth_data['instance_id'];
            $_SESSION['bookgram_page_uuid'] = $auth_data['page_uuid'];
            $_SESSION['bookgram_plugin_id'] = $auth_data['plugin_id'];

            // Set environment variables for plugin access
            if (!function_exists('get_instance_path')) {
                require_once __DIR__ . '/plugins.php';
            }

            $instance_path = get_instance_path(
                $auth_data['page_uuid'],
                $auth_data['plugin_id'],
                $auth_data['instance_id']
            );

            $_ENV['BOOKGRAM_INSTANCE_PATH'] = $instance_path;
            $_ENV['BOOKGRAM_INSTANCE_ID'] = $auth_data['instance_id'];
            $_ENV['BOOKGRAM_PAGE_UUID'] = $auth_data['page_uuid'];

            return $auth_data;
        }
    }

    // Check if already authenticated in this session (fallback)
    if (isset($_SESSION['bookgram_user']) && isset($_SESSION['bookgram_instance_id'])) {
        // Restore environment variables from session
        if (isset($_SESSION['bookgram_page_uuid']) &&
            isset($_SESSION['bookgram_plugin_id']) &&
            isset($_SESSION['bookgram_instance_id'])) {

            if (!function_exists('get_instance_path')) {
                require_once __DIR__ . '/plugins.php';
            }

            $instance_path = get_instance_path(
                $_SESSION['bookgram_page_uuid'],
                $_SESSION['bookgram_plugin_id'],
                $_SESSION['bookgram_instance_id']
            );

            $_ENV['BOOKGRAM_INSTANCE_PATH'] = $instance_path;
            $_ENV['BOOKGRAM_INSTANCE_ID'] = $_SESSION['bookgram_instance_id'];
            $_ENV['BOOKGRAM_PAGE_UUID'] = $_SESSION['bookgram_page_uuid'];
        }

        return [
            'username' => $_SESSION['bookgram_user'],
            'role' => $_SESSION['bookgram_role'],
            'instance_id' => $_SESSION['bookgram_instance_id'],
            'page_uuid' => $_SESSION['bookgram_page_uuid'] ?? null,
            'plugin_id' => $_SESSION['bookgram_plugin_id'] ?? null
        ];
    }

    return false;
}

/**
 * Check if current user is admin
 *
 * @return bool True if user is admin
 */
function bookgram_is_admin() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    return isset($_SESSION['bookgram_role']) && $_SESSION['bookgram_role'] === 'admin';
}

/**
 * Check if current user is editor or admin
 *
 * @return bool True if user is editor or admin
 */
function bookgram_is_editor() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    $role = $_SESSION['bookgram_role'] ?? 'guest';
    return $role === 'admin' || $role === 'editor';
}

/**
 * Check if user is logged in
 *
 * @return bool True if user is authenticated
 */
function bookgram_is_logged_in() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    return isset($_SESSION['bookgram_user']);
}

/**
 * Get current BookGram username
 *
 * @return string|null Username or null if not authenticated
 */
function bookgram_get_username() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    return $_SESSION['bookgram_user'] ?? null;
}

/**
 * Get current instance ID
 *
 * @return string|null Instance ID or null if not set
 */
function bookgram_get_instance_id() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    return $_SESSION['bookgram_instance_id'] ?? null;
}

/**
 * Get current instance path
 *
 * @return string|null Instance path or null if not set
 */
function bookgram_get_instance_path() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // Try to get from environment first
    if (isset($_ENV['BOOKGRAM_INSTANCE_PATH']) && $_ENV['BOOKGRAM_INSTANCE_PATH']) {
        return $_ENV['BOOKGRAM_INSTANCE_PATH'];
    }

    // Reconstruct from session data if not in environment
    if (isset($_SESSION['bookgram_page_uuid']) &&
        isset($_SESSION['bookgram_plugin_id']) &&
        isset($_SESSION['bookgram_instance_id'])) {

        // Load plugins.php to get the get_instance_path function
        if (!function_exists('get_instance_path')) {
            require_once __DIR__ . '/plugins.php';
        }

        $path = get_instance_path(
            $_SESSION['bookgram_page_uuid'],
            $_SESSION['bookgram_plugin_id'],
            $_SESSION['bookgram_instance_id']
        );

        // Cache it back to $_ENV for this request
        $_ENV['BOOKGRAM_INSTANCE_PATH'] = $path;

        return $path;
    }

    return null;
}

/**
 * Require authentication - die if not authenticated
 *
 * @param string $message Optional error message
 */
function bookgram_require_auth($message = null) {
    if (!bookgram_is_logged_in()) {
        if ($message === null) {
            $message = 'Authentication required. Please access this page via the BookGram article editor.';
        }

        die('<html><head><title>Authentication Required</title><style>
            body { font-family: "MS Sans Serif", sans-serif; padding: 40px; background: #008080; }
            .error-box { background: #ffffff; border: 2px solid #cc0000; padding: 20px; max-width: 600px; margin: 0 auto; box-shadow: 2px 2px 0 #000000; }
            .error-box h2 { color: #cc0000; margin-top: 0; }
            .error-box a { color: #0000ff; text-decoration: underline; }
        </style></head><body>
            <div class="error-box">
                <h2>⚠ Authentication Required</h2>
                <p>' . htmlspecialchars($message) . '</p>
                <p><a href="/">Return to Home</a></p>
            </div>
        </body></html>');
    }
}

/**
 * Require admin role - die if not admin
 *
 * @param string $message Optional error message
 */
function bookgram_require_admin($message = null) {
    bookgram_require_auth();

    if (!bookgram_is_admin()) {
        if ($message === null) {
            $message = 'Admin access required. You do not have permission to access this page.';
        }

        die('<html><head><title>Access Denied</title><style>
            body { font-family: "MS Sans Serif", sans-serif; padding: 40px; background: #008080; }
            .error-box { background: #ffffff; border: 2px solid #cc0000; padding: 20px; max-width: 600px; margin: 0 auto; box-shadow: 2px 2px 0 #000000; }
            .error-box h2 { color: #cc0000; margin-top: 0; }
            .error-box a { color: #0000ff; text-decoration: underline; }
        </style></head><body>
            <div class="error-box">
                <h2>⚠ Access Denied</h2>
                <p>' . htmlspecialchars($message) . '</p>
                <p><a href="/">Return to Home</a></p>
            </div>
        </body></html>');
    }
}

/**
 * Require editor role - die if not editor or admin
 *
 * @param string $message Optional error message
 */
function bookgram_require_editor($message = null) {
    bookgram_require_auth();

    if (!bookgram_is_editor()) {
        if ($message === null) {
            $message = 'Editor access required. You do not have permission to access this page.';
        }

        die('<html><head><title>Access Denied</title><style>
            body { font-family: "MS Sans Serif", sans-serif; padding: 40px; background: #008080; }
            .error-box { background: #ffffff; border: 2px solid #cc0000; padding: 20px; max-width: 600px; margin: 0 auto; box-shadow: 2px 2px 0 #000000; }
            .error-box h2 { color: #cc0000; margin-top: 0; }
            .error-box a { color: #0000ff; text-decoration: underline; }
        </style></head><body>
            <div class="error-box">
                <h2>⚠ Access Denied</h2>
                <p>' . htmlspecialchars($message) . '</p>
                <p><a href="/">Return to Home</a></p>
            </div>
        </body></html>');
    }
}

/**
 * HTML escape helper function
 * (Fallback if BookGram's e() function is not available)
 *
 * @param string $string String to escape
 * @return string Escaped string
 */
if (!function_exists('e')) {
    function e($string) {
        return htmlspecialchars((string)$string, ENT_QUOTES, 'UTF-8');
    }
}
