<?php
/**
 * PHP Embed Executor
 *
 * Executes PHP embed files in a restricted environment
 * Security: Function blacklist enforced via static analysis and runtime restrictions
 */

require_once __DIR__ . '/plugins.php';

/**
 * Execute a PHP embed file securely
 */
function execute_php_embed_secure() {
    // Get and validate file parameter
    $filepath = $_GET['file'] ?? '';

    if (empty($filepath)) {
        output_error('No file specified');
        return;
    }

    // Sanitize path
    $safe_filepath = sanitize_php_embed_path($filepath);
    if ($safe_filepath === false) {
        output_error('Invalid file path');
        return;
    }

    // Check file exists
    if (!php_embed_exists($safe_filepath)) {
        output_error('File not found');
        return;
    }

    // Scan for blacklisted functions
    $violations = scan_php_code_for_blacklisted_functions($safe_filepath);
    if (!empty($violations)) {
        $violation_list = implode(', ', array_column($violations, 'function'));
        output_error('Security violation: Forbidden functions detected: ' . $violation_list);
        return;
    }

    // Build full path
    $php_embeds_dir = dirname(__DIR__, 2) . '/php_embeds/';
    $full_path = $php_embeds_dir . $safe_filepath;

    // Extract parameters (excluding file parameter)
    $params = $_GET;
    unset($params['file']);

    // Set up execution environment
    setup_restricted_environment();

    // Execute with output buffering
    try {
        ob_start();

        // Make parameters available to the script
        extract($params, EXTR_SKIP);

        // Include the PHP file
        include $full_path;

        $output = ob_get_clean();

        // Output with security headers
        output_html($output);

    } catch (Throwable $e) {
        ob_end_clean();
        output_error('Execution error: ' . $e->getMessage());
    }
}

/**
 * Set up restricted PHP environment
 */
function setup_restricted_environment() {
    // Set strict error reporting
    error_reporting(E_ALL);
    ini_set('display_errors', '0');
    ini_set('log_errors', '1');

    // Set execution time limit (5 seconds max)
    @set_time_limit(5);

    // Memory limit
    @ini_set('memory_limit', '32M');

    // Disable file uploads
    @ini_set('file_uploads', '0');

    // Custom error handler
    set_error_handler(function($errno, $errstr, $errfile, $errline) {
        error_log("PHP Embed Error: $errstr in $errfile:$errline");
        // Don't output errors to user
        return true;
    });
}

/**
 * Output HTML with security headers
 */
function output_html($html) {
    header('Content-Type: text/html; charset=UTF-8');
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: SAMEORIGIN');
    header('Content-Security-Policy: default-src \'self\' \'unsafe-inline\'; script-src \'self\' \'unsafe-inline\' \'unsafe-eval\'');

    echo $html;
}

/**
 * Output error message
 */
function output_error($message) {
    header('Content-Type: text/html; charset=UTF-8');

    echo '<!DOCTYPE html>';
    echo '<html><head><title>Error</title></head><body>';
    echo '<div style="padding: 20px; background: #ffebee; border: 2px solid #c62828; margin: 20px;">';
    echo '<h3>PHP Embed Error</h3>';
    echo '<p>' . htmlspecialchars($message, ENT_QUOTES) . '</p>';
    echo '</div>';
    echo '</body></html>';
}

// Execute if called directly
if (basename($_SERVER['PHP_SELF']) === 'php_executor.php' ||
    (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], 'php-execute') !== false)) {
    execute_php_embed_secure();
}
