<?php
// app/core/functions.php

// --- Storage Paths ---
define('CONFIG_FILE', __DIR__ . '/../storage/config.json');
define('USERS_FILE', __DIR__ . '/../storage/users.json');
define('CONTENT_DIR', __DIR__ . '/../storage/pages/');
define('REVISIONS_DIR', __DIR__ . '/../storage/revisions/');
define('UPLOADS_DIR', __DIR__ . '/../../public/uploads/');
define('UUID_INDEX_FILE', __DIR__ . '/../storage/uuid_index.json');
define('HASHTAG_INDEX_FILE', __DIR__ . '/../storage/hashtags.json');

// --- FILE UPLOAD HANDLER ---

/**
 * Handles a file upload securely.
 * @param string $file_key The key from the $_FILES array (e.g., 'profile_photo').
 * @param string $upload_subdir The sub-directory in /public/uploads/ (e.g., 'avatars').
 * @return string|null The new web-accessible path (e.g., '/public/uploads/avatars/...') or null.
 */
function handle_image_upload($file_key, $upload_subdir) {
    // 1. Check if file was uploaded without errors
    if (!isset($_FILES[$file_key]) || $_FILES[$file_key]['error'] !== UPLOAD_ERR_OK) {
        // Common case: No file was submitted, return null without logging error
        if (isset($_FILES[$file_key]) && $_FILES[$file_key]['error'] === UPLOAD_ERR_NO_FILE) {
            return null;
        }
        // Log actual upload errors
        $error_code = $_FILES[$file_key]['error'] ?? 'Unknown';
        error_log("File upload error for key '$file_key': Code $error_code");
        return null;
    }

    $file = $_FILES[$file_key];
    $max_size = 5 * 1024 * 1024; // 5 MB

    // 2. Check file size
    if ($file['size'] > $max_size) {
        error_log("Upload failed: File too large for " . $file_key . " (Size: " . $file['size'] . ")");
        return null; // File is too large
    }
    if ($file['size'] === 0) {
        error_log("Upload failed: File is empty for " . $file_key);
        return null; // File is empty
    }


    // 3. Check file type (MIME type)
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
    if (!in_array($mime_type, $allowed_types)) {
        error_log("Upload failed: Invalid file type ($mime_type) for " . $file_key);
        return null; // Not an allowed image type
    }

    // 4. Create a unique filename
    $extension_map = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/gif' => 'gif'];
    $extension = $extension_map[$mime_type] ?? pathinfo($file['name'], PATHINFO_EXTENSION); // Fallback to original ext if needed
    if (empty($extension)) $extension = 'jpg'; // Default if still empty

    $unique_name = uniqid() . '-' . bin2hex(random_bytes(8)) . '.' . $extension;

    // 5. Create the destination directory
    $upload_path = UPLOADS_DIR . $upload_subdir . '/';
    if (!is_dir($upload_path)) {
        if (!mkdir($upload_path, 0755, true)) {
             error_log("Upload failed: Could not create directory " . $upload_path);
             return null; // Directory creation failed
        }
    }


    // 6. Move the file
    $destination = $upload_path . $unique_name;
    // Check if the uploaded file is valid before moving
    if (!is_uploaded_file($file['tmp_name'])) {
         error_log("Upload failed: Invalid upload file for " . $file_key);
         return null;
    }
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        // Return the web-accessible path
        return '/public/uploads/' . $upload_subdir . '/' . $unique_name;
    } else {
        error_log("Upload failed: Could not move uploaded file from " . $file['tmp_name'] . " to " . $destination);
    }

    return null; // Move failed
}

// --- Page Image Management Functions ---

/**
 * Get and ensure existence of page images directory
 * @param string $page_uuid The page UUID
 * @return string|false The directory path or false on failure
 */
function get_page_images_dir($page_uuid) {
    if (empty($page_uuid)) return false;

    // Sanitize UUID (allow dots since uniqid() includes them)
    $page_uuid = preg_replace('/[^a-zA-Z0-9_.\-]/', '', $page_uuid);

    $dir = __DIR__ . '/../storage/page_images/' . $page_uuid;

    if (!is_dir($dir)) {
        if (!mkdir($dir, 0755, true)) {
            error_log("Failed to create page images directory: " . $dir);
            return false;
        }
    }

    return $dir;
}

/**
 * Load image metadata for a page
 * @param string $page_uuid The page UUID
 * @return array Array of image metadata
 */
function load_page_images($page_uuid) {
    $dir = get_page_images_dir($page_uuid);
    if (!$dir) return [];

    $metadata_file = $dir . '/images.json';

    if (!file_exists($metadata_file)) {
        return [];
    }

    $content = @file_get_contents($metadata_file);
    if ($content === false) {
        error_log("Failed to read images metadata: " . $metadata_file);
        return [];
    }

    $data = json_decode($content, true);
    return is_array($data) ? $data : [];
}

/**
 * Save image metadata to JSON file
 * @param string $page_uuid The page UUID
 * @param array $images_array Array of image metadata
 * @return bool Success status
 */
function save_page_images_metadata($page_uuid, $images_array) {
    $dir = get_page_images_dir($page_uuid);
    if (!$dir) return false;

    $metadata_file = $dir . '/images.json';

    $content = json_encode($images_array, JSON_PRETTY_PRINT);
    if (@file_put_contents($metadata_file, $content, LOCK_EX) === false) {
        error_log("Failed to save images metadata: " . $metadata_file);
        return false;
    }

    return true;
}

/**
 * Resize uploaded image to reasonable web size
 * @param string $source_path Source image path
 * @param string $dest_path Destination image path
 * @param int $max_pixels Maximum pixels (default 2 megapixels)
 * @return array|false Array with width/height or false on failure
 */
function resize_uploaded_image($source_path, $dest_path, $max_pixels = 2000000) {
    // Get image info
    $image_info = @getimagesize($source_path);
    if (!$image_info) return false;

    list($width, $height, $type) = $image_info;
    $current_pixels = $width * $height;

    // Load image based on type
    switch ($type) {
        case IMAGETYPE_JPEG:
            $source = @imagecreatefromjpeg($source_path);
            break;
        case IMAGETYPE_PNG:
            $source = @imagecreatefrompng($source_path);
            break;
        case IMAGETYPE_GIF:
            $source = @imagecreatefromgif($source_path);
            break;
        default:
            return false;
    }

    if (!$source) return false;

    // Calculate new dimensions if needed
    if ($current_pixels > $max_pixels) {
        $scale = sqrt($max_pixels / $current_pixels);
        $new_width = (int)($width * $scale);
        $new_height = (int)($height * $scale);
    } else {
        $new_width = $width;
        $new_height = $height;
    }

    // Create destination image
    $dest_image = imagecreatetruecolor($new_width, $new_height);

    // Preserve transparency for PNG
    if ($type == IMAGETYPE_PNG) {
        imagealphablending($dest_image, false);
        imagesavealpha($dest_image, true);
    }

    imagecopyresampled($dest_image, $source, 0, 0, 0, 0,
                       $new_width, $new_height, $width, $height);

    // Determine output format based on destination extension
    $ext = strtolower(pathinfo($dest_path, PATHINFO_EXTENSION));

    if ($ext === 'png' && $type == IMAGETYPE_PNG) {
        $result = imagepng($dest_image, $dest_path, 9);
    } else {
        // Convert to JPEG for all other cases (smaller file size)
        $result = imagejpeg($dest_image, $dest_path, 85);
    }

    imagedestroy($source);
    imagedestroy($dest_image);

    return $result ? ['width' => $new_width, 'height' => $new_height] : false;
}

/**
 * Handle image upload for a specific page
 * @param string $page_uuid The page UUID
 * @param string $file_key The key from $_FILES array
 * @return array|null Image metadata or null on failure
 */
function handle_page_image_upload($page_uuid, $file_key) {
    $debug_log = __DIR__ . '/../../debug.log';
    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - handle_page_image_upload called with UUID: $page_uuid, key: $file_key\n", FILE_APPEND);

    // Validate file upload
    if (!isset($_FILES[$file_key]) || $_FILES[$file_key]['error'] !== UPLOAD_ERR_OK) {
        if (isset($_FILES[$file_key]) && $_FILES[$file_key]['error'] === UPLOAD_ERR_NO_FILE) {
            file_put_contents($debug_log, date('Y-m-d H:i:s') . " - No file uploaded\n", FILE_APPEND);
            return null;
        }
        $error_code = $_FILES[$file_key]['error'] ?? 'Unknown';
        file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Image upload error for key '$file_key': Code $error_code\n", FILE_APPEND);
        error_log("Image upload error for key '$file_key': Code " . $error_code);
        return null;
    }

    $file = $_FILES[$file_key];
    $max_size = 10 * 1024 * 1024; // 10 MB before resize

    // Check file size
    if ($file['size'] > $max_size) {
        file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Upload failed: File too large (Size: " . $file['size'] . ")\n", FILE_APPEND);
        error_log("Upload failed: File too large (Size: " . $file['size'] . ")");
        return null;
    }
    if ($file['size'] === 0) {
        file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Upload failed: File is empty\n", FILE_APPEND);
        error_log("Upload failed: File is empty");
        return null;
    }

    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - File size OK: " . $file['size'] . " bytes\n", FILE_APPEND);

    // Validate MIME type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - MIME type detected: $mime_type\n", FILE_APPEND);

    $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
    if (!in_array($mime_type, $allowed_types)) {
        file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Upload failed: Invalid file type ($mime_type)\n", FILE_APPEND);
        error_log("Upload failed: Invalid file type ($mime_type)");
        return null;
    }

    // Get page images directory
    $dir = get_page_images_dir($page_uuid);
    if (!$dir) {
        error_log("Upload failed: Could not create images directory for UUID: " . $page_uuid);
        return null;
    }

    // Generate unique filename
    $extension_map = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/gif' => 'gif'];
    $extension = $extension_map[$mime_type] ?? 'jpg';
    $unique_id = 'img_' . time() . '_' . bin2hex(random_bytes(4));
    $filename = $unique_id . '.' . $extension;
    $destination = $dir . '/' . $filename;

    // Validate uploaded file
    if (!is_uploaded_file($file['tmp_name'])) {
        error_log("Upload failed: Security check failed");
        return null;
    }

    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $destination)) {
        error_log("Upload failed: Could not move file to " . $destination);
        return null;
    }

    // Resize image
    $temp_dest = $destination . '.tmp';
    $dimensions = resize_uploaded_image($destination, $temp_dest);

    if ($dimensions) {
        // Replace original with resized version
        if (!@rename($temp_dest, $destination)) {
            error_log("Upload warning: Could not replace original with resized version");
            @unlink($temp_dest);
            // Continue with original file
            $image_info = getimagesize($destination);
            $dimensions = ['width' => $image_info[0], 'height' => $image_info[1]];
        }
    } else {
        // If resize failed, keep original but get dimensions
        $image_info = getimagesize($destination);
        $dimensions = ['width' => $image_info[0], 'height' => $image_info[1]];
    }

    // Return image metadata
    $metadata = [
        'id' => $unique_id,
        'filename' => $filename,
        'original_name' => basename($file['name']),
        'description' => '',
        'uploaded_at' => date('Y-m-d H:i:s'),
        'size' => filesize($destination),
        'width' => $dimensions['width'],
        'height' => $dimensions['height'],
        'uploaded_by' => $_SESSION['user'] ?? 'unknown'
    ];

    file_put_contents($debug_log, date('Y-m-d H:i:s') . " - Upload SUCCESS: " . $filename . " (" . $dimensions['width'] . "x" . $dimensions['height'] . ")\n", FILE_APPEND);

    return $metadata;
}

/**
 * Delete an image from a page
 * @param string $page_uuid The page UUID
 * @param string $image_id The image ID
 * @return bool Success status
 */
function delete_page_image($page_uuid, $image_id) {
    $images = load_page_images($page_uuid);
    $dir = get_page_images_dir($page_uuid);

    if (!$dir) return false;

    // Find and remove image from metadata
    $found = false;
    $updated_images = [];

    foreach ($images as $img) {
        if ($img['id'] === $image_id) {
            $found = true;
            // Delete physical file
            $file_path = $dir . '/' . $img['filename'];
            if (file_exists($file_path)) {
                if (!@unlink($file_path)) {
                    error_log("Failed to delete image file: " . $file_path);
                    return false;
                }
            }
        } else {
            $updated_images[] = $img;
        }
    }

    if (!$found) {
        error_log("Delete failed: Image ID not found: " . $image_id);
        return false;
    }

    // Save updated metadata
    return save_page_images_metadata($page_uuid, $updated_images);
}

/**
 * Update image description and optionally rename
 * @param string $page_uuid The page UUID
 * @param string $image_id The image ID
 * @param string $description The new description
 * @param string|null $filename Optional new display filename
 * @return bool Success status
 */
function update_page_image_metadata($page_uuid, $image_id, $description, $filename = null) {
    $images = load_page_images($page_uuid);
    $dir = get_page_images_dir($page_uuid);

    if (!$dir) return false;

    $found = false;

    foreach ($images as &$img) {
        if ($img['id'] === $image_id) {
            $found = true;
            $img['description'] = trim($description);

            // Optional: handle filename rename (not changing physical file, just display name)
            if ($filename !== null && !empty(trim($filename))) {
                $img['original_name'] = trim($filename);
            }

            break;
        }
    }
    unset($img); // Break reference

    if (!$found) {
        error_log("Update failed: Image ID not found: " . $image_id);
        return false;
    }

    return save_page_images_metadata($page_uuid, $images);
}

/**
 * Serve image file with security checks
 * @param string $page_uuid The page UUID
 * @param string $filename The image filename
 */
function serve_page_image($page_uuid, $filename) {
    // Sanitize inputs (allow dots since uniqid() includes them)
    $page_uuid = preg_replace('/[^a-zA-Z0-9_.\-]/', '', $page_uuid);
    $filename = basename($filename); // Prevent directory traversal

    // Validate page exists and user has access
    $uuid_index = load_uuid_index();
    $page_path = $uuid_index[$page_uuid] ?? null;

    if (!$page_path) {
        http_response_code(404);
        die('Image not found');
    }

    $page_data = get_page_data($page_path);
    if (!$page_data) {
        http_response_code(404);
        die('Image not found');
    }

    // Check if user can view this page
    if (!can_view_private_page($page_data)) {
        http_response_code(403);
        die('Access denied');
    }

    // Get image directory
    $dir = get_page_images_dir($page_uuid);
    if (!$dir) {
        http_response_code(404);
        die('Image not found');
    }

    $file_path = $dir . '/' . $filename;

    // Verify file exists and is within allowed directory
    $real_path = realpath($file_path);
    $real_dir = realpath($dir);

    if ($real_path === false || strpos($real_path, $real_dir) !== 0 || !is_file($real_path)) {
        http_response_code(404);
        die('Image not found');
    }

    // Determine MIME type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $real_path);
    finfo_close($finfo);

    // Serve file
    header('Content-Type: ' . $mime_type);
    header('Content-Length: ' . filesize($real_path));
    header('Cache-Control: public, max-age=31536000'); // Cache for 1 year
    header('Content-Disposition: inline; filename="' . basename($filename) . '"');

    readfile($real_path);
    exit;
}

// --- HTML Embed Functions ---

/**
 * Handle HTML file upload to embeds directory
 * @param string $file_key POST key for file upload
 * @return string|null Filename (not path) of uploaded file or null on failure
 */
function handle_html_upload($file_key) {
    if (!isset($_FILES[$file_key]) || $_FILES[$file_key]['error'] === UPLOAD_ERR_NO_FILE) {
        error_log("Upload failed: No file uploaded");
        return null;
    }

    $file = $_FILES[$file_key];

    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        error_log("Upload failed: Upload error code " . $file['error']);
        return null;
    }

    // Check file size (10MB max for HTML files)
    $max_size = 10 * 1024 * 1024; // 10MB in bytes
    if ($file['size'] > $max_size) {
        error_log("Upload failed: File size " . $file['size'] . " exceeds " . $max_size);
        return null;
    }

    if ($file['size'] === 0) {
        error_log("Upload failed: File is empty");
        return null;
    }

    // Validate MIME type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    $allowed_mimes = ['text/html', 'text/plain', 'application/octet-stream']; // Some servers report text/plain for HTML
    if (!in_array($mime_type, $allowed_mimes)) {
        error_log("Upload failed: Invalid MIME type " . $mime_type);
        return null;
    }

    // Validate file extension
    $original_name = basename($file['name']);
    $ext = strtolower(pathinfo($original_name, PATHINFO_EXTENSION));
    if (!in_array($ext, ['html', 'htm'])) {
        error_log("Upload failed: Invalid extension " . $ext);
        return null;
    }

    // Sanitize filename (alphanumeric, hyphens, underscores, dots only)
    $safe_name = preg_replace('/[^a-zA-Z0-9_.-]/', '_', pathinfo($original_name, PATHINFO_FILENAME));
    $filename = $safe_name . '.' . $ext;

    // Ensure embeds directory exists
    $embeds_dir = dirname(__DIR__, 2) . '/embeds/';
    if (!is_dir($embeds_dir)) {
        mkdir($embeds_dir, 0755, true);
    }

    // Check if file exists and create unique name if needed
    $destination = $embeds_dir . $filename;
    $counter = 1;
    while (file_exists($destination)) {
        $filename = $safe_name . '_' . $counter . '.' . $ext;
        $destination = $embeds_dir . $filename;
        $counter++;
    }

    // Validate that this is actually an uploaded file
    if (!is_uploaded_file($file['tmp_name'])) {
        error_log("Upload failed: Security check failed");
        return null;
    }

    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        error_log("HTML file uploaded successfully: " . $filename);
        return $filename;
    } else {
        error_log("Upload failed: Could not move uploaded file");
    }

    return null;
}

/**
 * Get list of all HTML files in embeds directory
 * @return array Array of embed files with metadata
 */
function get_html_embeds() {
    $embeds_dir = dirname(__DIR__, 2) . '/embeds/';
    $embeds = [];

    if (!is_dir($embeds_dir)) {
        return $embeds;
    }

    // Use RecursiveDirectoryIterator to scan directory and subdirectories
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($embeds_dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $ext = strtolower($file->getExtension());
            if (in_array($ext, ['html', 'htm'])) {
                // Get relative path from embeds directory
                $relative_path = str_replace($embeds_dir, '', $file->getPathname());

                // Skip hidden files and README
                if (strpos(basename($relative_path), '.') === 0 || basename($relative_path) === 'README.md') {
                    continue;
                }

                $embeds[] = [
                    'filename' => $relative_path,
                    'full_path' => $file->getPathname(),
                    'size' => $file->getSize(),
                    'modified' => $file->getMTime()
                ];
            }
        }
    }

    // Sort alphabetically by filename
    usort($embeds, function($a, $b) {
        return strcmp($a['filename'], $b['filename']);
    });

    return $embeds;
}

/**
 * Delete HTML embed file
 * @param string $filename Filename relative to embeds directory
 * @return bool Success/failure
 */
function delete_html_embed($filename) {
    // Sanitize filename
    $filename = str_replace('..', '', $filename);
    $filename = trim($filename, '/');

    // Validate extension
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    if (!in_array($ext, ['html', 'htm'])) {
        error_log("Delete failed: Invalid extension");
        return false;
    }

    // Build full path
    $embeds_dir = dirname(__DIR__, 2) . '/embeds/';
    $full_path = $embeds_dir . $filename;

    // Verify file is within embeds directory (prevent directory traversal)
    $real_path = realpath($full_path);
    $real_embeds_dir = realpath($embeds_dir);

    if ($real_path === false || strpos($real_path, $real_embeds_dir) !== 0) {
        error_log("Delete failed: File not in embeds directory");
        return false;
    }

    // Check file exists
    if (!file_exists($full_path) || !is_file($full_path)) {
        error_log("Delete failed: File not found");
        return false;
    }

    // Delete file
    if (unlink($full_path)) {
        error_log("Deleted HTML embed: " . $filename);
        return true;
    } else {
        error_log("Delete failed: Could not delete file");
        return false;
    }
}

/**
 * Format bytes to human-readable size
 * @param int $bytes File size in bytes
 * @return string Formatted size (e.g., "1.5 MB")
 */
function format_bytes($bytes) {
    if ($bytes == 0) {
        return '0 B';
    }

    $units = ['B', 'KB', 'MB', 'GB'];
    $factor = floor((strlen((string)$bytes) - 1) / 3);
    $factor = min($factor, count($units) - 1);

    return sprintf("%.1f %s", $bytes / pow(1024, $factor), $units[$factor]);
}

// --- PHP Embed Functions ---

/**
 * Handle PHP file upload to php_embeds directory with security scanning
 * @param string $file_key POST key for file upload
 * @return string|null Filename (not path) of uploaded file or null on failure
 */
function handle_php_upload($file_key) {
    require_once __DIR__ . '/plugins.php';

    if (!isset($_FILES[$file_key]) || $_FILES[$file_key]['error'] === UPLOAD_ERR_NO_FILE) {
        error_log("Upload failed: No file uploaded");
        return null;
    }

    $file = $_FILES[$file_key];

    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        error_log("Upload failed: Upload error code " . $file['error']);
        return null;
    }

    // Check file size (5MB max for PHP files - smaller than HTML for security)
    $max_size = 5 * 1024 * 1024; // 5MB in bytes
    if ($file['size'] > $max_size) {
        error_log("Upload failed: File size " . $file['size'] . " exceeds " . $max_size);
        $_SESSION['admin_error'] = 'File too large. Maximum size is 5MB.';
        return null;
    }

    if ($file['size'] === 0) {
        error_log("Upload failed: File is empty");
        $_SESSION['admin_error'] = 'File is empty.';
        return null;
    }

    // Validate MIME type (must be text/plain or application/x-php)
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    $allowed_mimes = ['text/plain', 'text/x-php', 'application/x-httpd-php', 'application/x-php'];
    if (!in_array($mime_type, $allowed_mimes)) {
        error_log("Upload failed: Invalid MIME type " . $mime_type);
        $_SESSION['admin_error'] = 'Invalid file type. Only PHP files allowed.';
        return null;
    }

    // Validate file extension
    $original_name = basename($file['name']);
    $ext = strtolower(pathinfo($original_name, PATHINFO_EXTENSION));

    if ($ext !== 'php') {
        error_log("Upload failed: Invalid extension " . $ext);
        $_SESSION['admin_error'] = 'Invalid file extension. Only .php files allowed.';
        return null;
    }

    // Sanitize filename
    $safe_name = preg_replace('/[^a-zA-Z0-9_-]/', '_', pathinfo($original_name, PATHINFO_FILENAME));
    if (empty($safe_name)) {
        $safe_name = 'upload_' . time();
    }
    $filename = $safe_name . '.php';

    // Get php_embeds directory
    $php_embeds_dir = dirname(__DIR__, 2) . '/php_embeds/';

    // CRITICAL: Scan file for blacklisted functions BEFORE saving
    $temp_content = file_get_contents($file['tmp_name']);
    $blacklist = get_php_function_blacklist();
    $blacklist_lower = array_map('strtolower', $blacklist);
    $violations = [];

    // Tokenize and scan
    $tokens = token_get_all($temp_content);

    // Language constructs that need special detection
    $language_constructs = [
        T_EVAL => 'eval',
        T_INCLUDE => 'include',
        T_INCLUDE_ONCE => 'include_once',
        T_REQUIRE => 'require',
        T_REQUIRE_ONCE => 'require_once'
    ];

    for ($i = 0; $i < count($tokens); $i++) {
        if (!is_array($tokens[$i])) {
            continue;
        }

        $token_type = $tokens[$i][0];
        $line_number = $tokens[$i][2];

        // Check for language constructs
        if (isset($language_constructs[$token_type])) {
            $construct_name = $language_constructs[$token_type];
            if (in_array($construct_name, $blacklist_lower)) {
                $violations[] = [
                    'function' => $construct_name,
                    'line' => $line_number
                ];
            }
            continue;
        }

        // Check for regular function calls
        if ($token_type === T_STRING) {
            $function_name = $tokens[$i][1];

            // Check if next non-whitespace token is '('
            $j = $i + 1;
            while ($j < count($tokens) && is_array($tokens[$j]) && $tokens[$j][0] === T_WHITESPACE) {
                $j++;
            }

            if ($j < count($tokens) && $tokens[$j] === '(') {
                // This is a function call
                if (in_array(strtolower($function_name), $blacklist_lower)) {
                    $violations[] = [
                        'function' => $function_name,
                        'line' => $line_number
                    ];
                }
            }
        }
    }

    // If violations found, reject upload
    if (!empty($violations)) {
        $violation_list = implode(', ', array_unique(array_column($violations, 'function')));
        error_log("Upload failed: Security violations detected - " . $violation_list);
        $_SESSION['admin_error'] = 'Security Error: File contains forbidden functions: ' . $violation_list . '. These functions are blocked for security reasons.';
        return null;
    }

    // Check if file exists and create unique name if needed
    $destination = $php_embeds_dir . $filename;
    $counter = 1;
    while (file_exists($destination)) {
        $filename = $safe_name . '_' . $counter . '.php';
        $destination = $php_embeds_dir . $filename;
        $counter++;
    }

    // Validate that this is actually an uploaded file
    if (!is_uploaded_file($file['tmp_name'])) {
        error_log("Upload failed: Security check failed");
        $_SESSION['admin_error'] = 'Security check failed.';
        return null;
    }

    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        error_log("PHP file uploaded successfully: " . $filename);
        return $filename;
    } else {
        error_log("Upload failed: Could not move uploaded file");
        $_SESSION['admin_error'] = 'Failed to save file.';
    }

    return null;
}

/**
 * Get list of all PHP files in php_embeds directory
 * @return array Array of embed files with metadata
 */
function get_php_embeds() {
    $php_embeds_dir = dirname(__DIR__, 2) . '/php_embeds/';
    $embeds = [];

    if (!is_dir($php_embeds_dir)) {
        return $embeds;
    }

    // Scan directory for PHP files
    $files = scandir($php_embeds_dir);

    foreach ($files as $file) {
        if ($file === '.' || $file === '..') {
            continue;
        }

        $full_path = $php_embeds_dir . $file;

        if (is_file($full_path)) {
            $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            if ($ext === 'php') {
                // Skip hidden files and system files
                if (strpos($file, '.') === 0 || strpos($file, '_security_test') === 0) {
                    continue;
                }

                $embeds[] = [
                    'filename' => $file,
                    'full_path' => $full_path,
                    'size' => filesize($full_path),
                    'modified' => filemtime($full_path)
                ];
            }
        }
    }

    // Sort alphabetically by filename
    usort($embeds, function($a, $b) {
        return strcmp($a['filename'], $b['filename']);
    });

    return $embeds;
}

/**
 * Delete PHP embed file
 * @param string $filename Filename in php_embeds directory
 * @return bool Success/failure
 */
function delete_php_embed($filename) {
    // Sanitize filename
    $filename = basename($filename); // Only filename, no paths
    $filename = str_replace('..', '', $filename);

    // Validate extension
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    if ($ext !== 'php') {
        error_log("Delete failed: Invalid extension");
        return false;
    }

    // Prevent deletion of critical files
    if (strpos($filename, '_security_test') === 0) {
        error_log("Delete failed: Cannot delete test files");
        return false;
    }

    // Build full path
    $php_embeds_dir = dirname(__DIR__, 2) . '/php_embeds/';
    $full_path = $php_embeds_dir . $filename;

    // Verify file is within php_embeds directory (prevent directory traversal)
    $real_path = realpath($full_path);
    $real_php_embeds_dir = realpath($php_embeds_dir);

    if ($real_path === false || strpos($real_path, $real_php_embeds_dir) !== 0) {
        error_log("Delete failed: File not in php_embeds directory");
        return false;
    }

    // Check file exists
    if (!file_exists($full_path) || !is_file($full_path)) {
        error_log("Delete failed: File not found");
        return false;
    }

    // Delete file
    if (unlink($full_path)) {
        error_log("Deleted PHP embed: " . $filename);
        return true;
    } else {
        error_log("Delete failed: Could not delete file");
        return false;
    }
}

// --- Config Functions ---

function load_config() {
    if (!file_exists(CONFIG_FILE)) {
        if (basename($_SERVER['PHP_SELF']) !== 'setup.php') {
             die('Error: config.json not found. Please run setup.php');
        }
        return [];
    }
    $content = @file_get_contents(CONFIG_FILE);
    if ($content === false) {
        error_log("Failed to read config file: " . CONFIG_FILE);
        return [];
    }
    return json_decode($content, true) ?: [];
}


function save_config($config_data) {
    if (@file_put_contents(CONFIG_FILE, json_encode($config_data, JSON_PRETTY_PRINT), LOCK_EX) === false) {
        error_log("Failed to write config file: " . CONFIG_FILE);
        return false;
    }
    return true;
}

// --- User Data Functions ---

function load_users() {
    if (!file_exists(USERS_FILE)) {
         if (basename($_SERVER['PHP_SELF']) !== 'setup.php') {
             error_log("Users file not found: " . USERS_FILE);
         }
        return [];
    }
    $content = @file_get_contents(USERS_FILE);
     if ($content === false) {
        error_log("Failed to read users file: " . USERS_FILE);
        return [];
    }
    return json_decode($content, true) ?: [];
}


function save_users($users_data) {
     if (@file_put_contents(USERS_FILE, json_encode($users_data, JSON_PRETTY_PRINT), LOCK_EX) === false) {
         error_log("Failed to write users file: " . USERS_FILE);
         return false;
     }
     return true;
}

function update_user_profile($username, $tagline, $photo_url, $bio) {
    $users = load_users();

    if (!isset($users[$username])) {
        return false;
    }

    $users[$username]['tagline'] = $tagline;
    $users[$username]['photo_url'] = $photo_url;
    $users[$username]['bio'] = $bio;

    return save_users($users);
}


// --- UTILITY FUNCTIONS ---

function sanitize_path($path) {
    $path = trim($path, '/');
    $path = str_replace('..', '', $path);
    $parts = explode('/', $path);
    $safe_parts = [];
    foreach ($parts as $part) {
        if ($part !== '' && $part[0] !== '.') {
            $safe_parts[] = $part;
        }
    }
    $path = implode('/', $safe_parts);
    $path = preg_replace('/\/+/', '/', $path);
    return $path;
}

function path_to_flat_slug($path) {
    $slug = str_replace('/', '_', $path);
    $slug = preg_replace('/[^a-zA-Z0-9_-]/', '-', $slug);
    $slug = trim(preg_replace('/-+/', '-', $slug), '-');
    return $slug ?: 'invalid-path';
}


// --- UUID Management Functions ---

function generate_page_uuid() {
    return uniqid('page_', true) . '-' . bin2hex(random_bytes(8));
}

function generate_share_token() {
    return bin2hex(random_bytes(16));
}

/**
 * Check if current user can view a private page
 * @param array $page_data The page data
 * @return bool True if user can view the page
 */
function can_view_private_page($page_data) {
    // Public pages are always viewable
    if (empty($page_data['is_private'])) {
        return true;
    }

    // Admins can view all pages
    if (is_admin()) {
        return true;
    }

    // Authors can view their own pages
    if (is_logged_in() && isset($page_data['author'])) {
        $current_user = get_current_username();
        if ($current_user === $page_data['author']) {
            return true;
        }
    }

    // Check if share token is provided in the URL
    if (isset($_GET['token']) && !empty($page_data['share_token'])) {
        if (hash_equals($page_data['share_token'], $_GET['token'])) {
            // Token is valid, but check if password is also required
            if (!empty($page_data['password_hash'])) {
                // Password required - check session
                if (session_status() === PHP_SESSION_NONE) {
                    @session_start();
                }
                $page_uuid = $page_data['uuid'] ?? null;
                if ($page_uuid && isset($_SESSION['authenticated_pages'][$page_uuid])) {
                    return true; // Password already verified
                }
                return false; // Need password verification
            }
            return true; // Token valid, no password required
        }
    }

    return false;
}

/**
 * Verify password for a private page
 * @param array $page_data The page data
 * @param string $password The password to verify
 * @return bool True if password is correct
 */
function verify_page_password($page_data, $password) {
    if (empty($page_data['password_hash'])) {
        return true; // No password set
    }

    if (password_verify($password, $page_data['password_hash'])) {
        // Store authentication in session
        if (session_status() === PHP_SESSION_NONE) {
            @session_start();
        }
        if (!isset($_SESSION['authenticated_pages'])) {
            $_SESSION['authenticated_pages'] = [];
        }
        $page_uuid = $page_data['uuid'] ?? null;
        if ($page_uuid) {
            $_SESSION['authenticated_pages'][$page_uuid] = time();
        }
        return true;
    }

    return false;
}

function load_uuid_index() {
    if (!file_exists(UUID_INDEX_FILE)) {
        return [];
    }
    $content = @file_get_contents(UUID_INDEX_FILE);
    if ($content === false) {
        error_log("Failed to read UUID index file: " . UUID_INDEX_FILE);
        return [];
    }
    return json_decode($content, true) ?: [];
}

function save_uuid_index($index) {
    ksort($index);
    $content = json_encode($index, JSON_PRETTY_PRINT);
    if (@file_put_contents(UUID_INDEX_FILE, $content, LOCK_EX) === false) {
         error_log("Failed to write UUID index file: " . UUID_INDEX_FILE);
         return false;
    }
    return true;
}

function update_uuid_index($uuid, $path) {
    if (empty($uuid)) return;

    $index = load_uuid_index();
    if ($path === null) {
        unset($index[$uuid]);
    } else {
        $index[$uuid] = $path;
    }
    save_uuid_index($index);
}


// --- Page & Revision Functions ---

function get_page_data($path) {
    $path = sanitize_path($path);
    $file = CONTENT_DIR . $path . '.json';

    if (empty($path) || !file_exists($file)) {
        return false;
    }

    $content = @file_get_contents($file);
    if ($content === false) {
        error_log("Failed to read page data file: " . $file);
        return false;
    }
    $data = json_decode($content, true);

    if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
        error_log("Failed to decode JSON for page: " . $path . " - Error: " . json_last_error_msg());
        return false;
    }
     if (!is_array($data)) {
        $data = [];
    }

    $needs_save = false;
    if (empty($data['uuid'])) {
        $data['uuid'] = generate_page_uuid();
        $needs_save = true;
    }

    $index = load_uuid_index();
    if (!isset($index[$data['uuid']]) || $index[$data['uuid']] !== $path) {
         update_uuid_index($data['uuid'], $path);
    }

    if ($needs_save && is_writable($file)) {
        file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT), LOCK_EX);
    } elseif ($needs_save) {
        error_log("Could not save UUID back to read-only file: " . $file);
    }

    $data['path'] = $path;
    $data['slug'] = basename($path);
    return $data;
}

function increment_and_get_page_data($path) {
    $page_data = get_page_data($path);
    if (!$page_data) {
        return false;
    }

    $save_needed = false;

    $current_hits = $page_data['hits'] ?? 0;
    $page_data['hits'] = $current_hits + 1;
    $save_needed = true;

    if (session_status() === PHP_SESSION_NONE) {
        @session_start(); // Use @ to suppress warning if already started elsewhere
    }
    if (!isset($_SESSION['viewed_pages'])) {
        $_SESSION['viewed_pages'] = [];
    }

    $page_uuid = $page_data['uuid'] ?? null;
    if ($page_uuid && !in_array($page_uuid, $_SESSION['viewed_pages'])) {
         $current_unique_hits = $page_data['unique_hits'] ?? 0;
         $page_data['unique_hits'] = $current_unique_hits + 1;
         $_SESSION['viewed_pages'][] = $page_uuid;
         $save_needed = true;
    }

    if ($save_needed) {
        $data_to_save = $page_data;
        unset($data_to_save['path'], $data_to_save['slug']);
        $content_file = CONTENT_DIR . $path . '.json';
        if (is_writable($content_file)) {
             file_put_contents($content_file, json_encode($data_to_save, JSON_PRETTY_PRINT), LOCK_EX);
        } else {
             error_log("Could not save hit count to read-only file: " . $content_file);
        }
    }

    return $page_data;
}


function get_revision_content($path, $revision_id) {
    $flat_slug = path_to_flat_slug($path);
    $file = REVISIONS_DIR . $flat_slug . '-' . $revision_id . '.txt';
    if (!file_exists($file)) {
        error_log("Revision file not found: " . $file);
        return 'Error: Revision file not found.';
    }
     $content = @file_get_contents($file);
     if ($content === false) {
         error_log("Failed to read revision file: " . $file);
         return 'Error: Could not read revision content.';
     }
    return $content;
}

function save_page_content($path, $title, $content, $username, $feature_image, $is_private = false, $password = null) {
    $path = sanitize_path($path);
    if (empty($path)) {
        error_log("Save failed: Path is empty.");
        return false;
    }

    $rev_id = time();
    $content_file = CONTENT_DIR . $path . '.json';
    $is_new_page = !file_exists($content_file);

    $page_data = get_page_data($path);

     if ($page_data === false && !$is_new_page) {
         error_log("Save failed: Could not read existing page data for: " . $path);
         return false;
     }

    if ($is_new_page) {
        $page_data = [
            'title' => $title,
            'author' => $username,
            'history' => [],
            'uuid' => generate_page_uuid(),
            'hits' => 0,
            'unique_hits' => 0,
            'is_private' => false,
            'share_token' => '',
            'password_hash' => ''
        ];
        update_uuid_index($page_data['uuid'], $path);
    }
     $page_data['history'] = $page_data['history'] ?? [];
     $page_data['uuid'] = $page_data['uuid'] ?? generate_page_uuid(); // Assign if somehow missing

    // Handle privacy settings
    $page_data['is_private'] = $is_private;
    if ($is_private && empty($page_data['share_token'])) {
        // Generate share token if page is being made private for the first time
        $page_data['share_token'] = generate_share_token();
    } elseif (!$is_private) {
        // Clear share token and password if page is made public
        $page_data['share_token'] = '';
        $page_data['password_hash'] = '';
    }

    // Handle password protection (optional, additional layer for private pages)
    if ($is_private && $password !== null && trim($password) !== '') {
        // Hash the password if provided
        $page_data['password_hash'] = password_hash(trim($password), PASSWORD_DEFAULT);
    } elseif ($is_private && $password === '') {
        // Empty password means remove password protection
        $page_data['password_hash'] = '';
    }
    // If not private, password_hash is already cleared above

    $flat_slug = path_to_flat_slug($path);
    $revision_file = REVISIONS_DIR . $flat_slug . '-' . $rev_id . '.txt';
    if (@file_put_contents($revision_file, $content, LOCK_EX) === false) {
        error_log("Failed to write revision file: " . $revision_file);
        return false;
    }

    // Extract hashtags from content
    $hashtags = extract_hashtags($content);
    $page_data['hashtags'] = $hashtags;

    // Update hashtag index
    update_page_hashtags($page_data['uuid'], $hashtags);

    $page_data['title'] = $title;
    $page_data['feature_image'] = $feature_image;
    $page_data['current_revision_id'] = (string)$rev_id;
    $page_data['last_edited_by'] = $username;
    array_unshift($page_data['history'], (string)$rev_id);

    $directory = dirname($content_file);
    if (!is_dir($directory)) {
        if (!mkdir($directory, 0755, true)) {
            error_log("Failed to create directory: " . $directory);
            return false;
        }
    }

    $save_data = $page_data;
    unset($save_data['path'], $save_data['slug']);
    if (@file_put_contents($content_file, json_encode($save_data, JSON_PRETTY_PRINT), LOCK_EX) === false) {
         error_log("Failed to write content file: " . $content_file);
         return false;
    }

    return true;
}


function delete_revision($path, $revision_id) {
    $path = sanitize_path($path);
    $page_data = get_page_data($path);

    if (!$page_data || ($page_data['current_revision_id'] ?? null) == $revision_id) {
        return false;
    }
     $history = $page_data['history'] ?? [];
     if (!is_array($history)) {
         error_log("Delete revision failed: History is not an array for page " . $path);
         return false;
     }

    $flat_slug = path_to_flat_slug($path);
    $revision_file = REVISIONS_DIR . $flat_slug . '-' . $revision_id . '.txt';
    if (file_exists($revision_file)) {
        if (!@unlink($revision_file)) {
             error_log("Failed to delete revision file: " . $revision_file);
             // return false; // Decide if failure to delete should stop JSON update
        }
    }

    $key = array_search((string)$revision_id, array_map('strval', $history));
    if ($key !== false) {
        unset($page_data['history'][$key]);
        $page_data['history'] = array_values($page_data['history']);
    }

    unset($page_data['path'], $page_data['slug']);
    $content_file = CONTENT_DIR . $path . '.json';
    if (!@file_put_contents($content_file, json_encode($page_data, JSON_PRETTY_PRINT), LOCK_EX)) {
         error_log("Failed to update page data after deleting revision for: " . $path);
         return false;
    }

    return true;
}


function delete_page($path) {
    $path = sanitize_path($path);
    $content_file = CONTENT_DIR . $path . '.json';

    $uuid_to_remove = null;
    if (file_exists($content_file)) {
        $content = @file_get_contents($content_file);
        if ($content !== false) {
             $data = json_decode($content, true);
             if (is_array($data) && isset($data['uuid'])) {
                 $uuid_to_remove = $data['uuid'];
             }
        } else {
             error_log("Could not read file before deleting: " . $content_file);
        }

        if (!@unlink($content_file)) {
            error_log("Failed to delete page file: " . $content_file);
            return false;
        }
    } else {
        return false; // Page doesn't exist
    }

    $flat_slug = path_to_flat_slug($path);
    $revision_files = glob(REVISIONS_DIR . $flat_slug . '-*.txt');
    foreach ($revision_files as $file) {
        @unlink($file);
    }

    if ($uuid_to_remove) {
        update_uuid_index($uuid_to_remove, null);

        // Clean up plugin instances associated with this page
        $instance_dir = __DIR__ . '/../storage/plugin_instances/' . $uuid_to_remove;
        if (is_dir($instance_dir)) {
            // Recursive delete function
            $delete_recursive = function($dir) use (&$delete_recursive) {
                if (!is_dir($dir)) {
                    return @unlink($dir);
                }

                $files = array_diff(scandir($dir), ['.', '..']);
                foreach ($files as $file) {
                    $path = $dir . '/' . $file;
                    is_dir($path) ? $delete_recursive($path) : @unlink($path);
                }

                return @rmdir($dir);
            };

            $delete_recursive($instance_dir);
        }

        // Clean up page images
        $images_dir = __DIR__ . '/../storage/page_images/' . $uuid_to_remove;
        if (is_dir($images_dir)) {
            $delete_recursive = function($dir) use (&$delete_recursive) {
                if (!is_dir($dir)) {
                    return @unlink($dir);
                }

                $files = array_diff(scandir($dir), ['.', '..']);
                foreach ($files as $file) {
                    $path = $dir . '/' . $file;
                    is_dir($path) ? $delete_recursive($path) : @unlink($path);
                }

                return @rmdir($dir);
            };

            $delete_recursive($images_dir);
        }
    }

    return true;
}


// --- DIRECTORY & SEARCH FUNCTIONS ---

function get_directory_contents($path) {
    $path = sanitize_path($path);
    $full_path = rtrim(CONTENT_DIR . $path, '/');
    $items = [];

    if (!is_dir(CONTENT_DIR)) {
         if (!mkdir(CONTENT_DIR, 0755, true)) {
             error_log("Failed to create base content directory: " . CONTENT_DIR);
             return ['folders' => [], 'pages' => []];
         }
    }

    if (!is_dir($full_path)) {
        return ['folders' => [], 'pages' => []];
    }

    $scan = @scandir($full_path);
    if ($scan === false) {
        error_log("Failed to scan directory: " . $full_path);
        return ['folders' => [], 'pages' => []];
    }

    $folders = [];
    $pages = [];

    foreach ($scan as $item) {
        if ($item === '.' || $item === '..') {
            continue;
        }

        $item_full_path = $full_path . '/' . $item;
        $item_rel_path = ($path ? $path . '/' : '') . $item;

        if (@is_dir($item_full_path)) {
            $folders[] = [
                'name' => $item,
                'path' => $item_rel_path,
            ];
        } elseif (@is_file($item_full_path) && pathinfo($item, PATHINFO_EXTENSION) === 'json') {
             $content = @file_get_contents($item_full_path);
             if ($content === false) {
                 error_log("Failed to read page file in get_directory_contents: " . $item_full_path);
                 continue;
             }
             $page_data = json_decode($content, true);
             if (!is_array($page_data)) { $page_data = []; }

            $page_path = substr($item_rel_path, 0, -5);

            // Skip private pages unless user can view them
            if (!empty($page_data['is_private']) && !can_view_private_page($page_data)) {
                continue;
            }

            $pages[] = [
                'name' => basename($item, '.json'),
                'path' => $page_path,
                'title' => $page_data['title'] ?? basename($item, '.json'),
                // Optionally add feature image here if needed for browser view performance
                // 'feature_image' => $page_data['feature_image'] ?? null
            ];
        }
    }

    usort($folders, function($a, $b) { return strcasecmp($a['name'], $b['name']); });
    usort($pages, function($a, $b) { return strcasecmp($a['title'], $b['title']); });

    return ['folders' => $folders, 'pages' => $pages];
}


function create_folder($current_path, $folder_name) {
    $current_path = sanitize_path($current_path);
    $folder_name_base = preg_replace('/[^a-zA-Z0-9_-]/', '-', basename($folder_name));
    $folder_name_base = trim(preg_replace('/-+/', '-', $folder_name_base), '-');

    if(empty($folder_name_base)) {
        error_log("Create folder failed: Invalid folder name after sanitization for " . $folder_name);
        return false;
    }

    $new_path = CONTENT_DIR . ($current_path ? $current_path . '/' : '') . $folder_name_base;

    if (file_exists($new_path)) {
        return false;
    }

    if (!@mkdir($new_path, 0755, true)) {
         $error = error_get_last();
         error_log("Create folder failed: Could not create directory $new_path. Error: " . ($error['message'] ?? 'Unknown error'));
         return false;
    }

    return true;
}


function delete_folder_recursive($path) {
    $path = sanitize_path($path);
    $dir = CONTENT_DIR . $path;

    if (!is_dir($dir)) { return false; }

    try {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS | FilesystemIterator::UNIX_PATHS),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $item) {
             if (!$item->isReadable()) {
                 error_log("Delete folder skipped unreadable item: " . $item->getPathname());
                 continue;
             }
             $item_path_rel = $path . '/' . $iterator->getSubPathName();

            if ($item->isDir()){
                @rmdir($item->getRealPath());
            } else {
                if ($item->isFile() && $item->getExtension() === 'json') {
                     $page_path_rel = substr($item_path_rel, 0, -5);
                     delete_page($page_path_rel); // Handles revisions and index
                } else {
                     @unlink($item->getRealPath());
                }
            }
        }
     } catch (UnexpectedValueException $e) {
         error_log("Error during directory iteration in delete_folder_recursive: " . $e->getMessage());
         // Attempt to remove the top dir anyway, might fail if iteration stopped early
     } catch (Exception $e) {
         error_log("General error during delete_folder_recursive: " . $e->getMessage());
         // return false; // Fail hard on general errors?
     }

    return @rmdir($dir);
}


function rename_item($old_path, $new_name) {
    $old_path = sanitize_path($old_path);
    $new_name_base = preg_replace('/[^a-zA-Z0-9_-]/', '-', basename($new_name));
    $new_name_base = trim(preg_replace('/-+/', '-', $new_name_base), '-');
     if (empty($new_name_base)) {
         error_log("Rename failed: Invalid new name after sanitization for " . $new_name);
         return false;
     }

    $parent_dir = dirname($old_path);
    $new_path = ($parent_dir === '.' ? '' : $parent_dir . '/') . $new_name_base;

    $old_full = CONTENT_DIR . $old_path;
    $new_full = CONTENT_DIR . $new_path;

    // Only skip rename if paths are EXACTLY the same (allow case-only renames on Linux)
    if ($old_full === $new_full) {
        return true;
    }

    if (file_exists($old_full . '.json')) {
        $old_full_file = $old_full . '.json';
        $new_full_file = $new_full . '.json';

        if (file_exists($new_full_file)) {
            error_log("Rename failed: Destination file exists: " . $new_full_file);
            return false;
        }

        $page_data = get_page_data($old_path);
        if ($page_data === false) {
             error_log("Rename failed: Could not read source page data for: " . $old_path);
             return false;
        }
        $uuid_to_update = $page_data['uuid'] ?? null;
         if (!$uuid_to_update) {
             error_log("Rename failed: Could not get UUID for page: " . $old_path);
         }

        if (!@rename($old_full_file, $new_full_file)) {
             error_log("Rename failed: Could not rename file from " . $old_full_file . " to " . $new_full_file);
             return false;
        }

        $old_flat_slug = path_to_flat_slug($old_path);
        $new_flat_slug = path_to_flat_slug($new_path);
        $revision_files = glob(REVISIONS_DIR . $old_flat_slug . '-*.txt');
        foreach ($revision_files as $file) {
            $new_rev_name = str_replace($old_flat_slug, $new_flat_slug, basename($file));
            @rename($file, REVISIONS_DIR . $new_rev_name);
        }

        if ($uuid_to_update) {
            update_uuid_index($uuid_to_update, $new_path);
        }
        return true;

    }
    elseif (is_dir($old_full)) {
        if (file_exists($new_full)) {
             error_log("Rename failed: Destination directory exists: " . $new_full);
             return false;
        }

        if (@rename($old_full, $new_full)) {
            // Update UUID index for all files in the folder
            $index = load_uuid_index();
            $changed = false;
            $old_path_prefix = rtrim($old_path, '/') . '/';
            $new_path_prefix = rtrim($new_path, '/') . '/';

            foreach ($index as $uuid => $current_path) {
                 if (strpos($current_path, $old_path_prefix) === 0) {
                     $updated_path = $new_path_prefix . substr($current_path, strlen($old_path_prefix));
                     $index[$uuid] = $updated_path;
                     $changed = true;
                }
                 else if ($current_path === $old_path) {
                    error_log("Warning: UUID index points directly to renamed folder: " . $old_path);
                 }
            }
            if ($changed) {
                save_uuid_index($index);
            }

            // Rename all revision files for pages in the renamed folder
            $old_flat_slug = path_to_flat_slug($old_path);
            $new_flat_slug = path_to_flat_slug($new_path);
            // Find all revision files that start with the old folder's flat slug
            $revision_files = glob(REVISIONS_DIR . $old_flat_slug . '_*.txt');
            if ($revision_files) {
                foreach ($revision_files as $file) {
                    // Replace the old folder slug with the new one
                    $basename = basename($file);
                    if (strpos($basename, $old_flat_slug . '_') === 0) {
                        $new_basename = $new_flat_slug . substr($basename, strlen($old_flat_slug));
                        @rename($file, REVISIONS_DIR . $new_basename);
                    }
                }
            }

            return true;
        } else {
             $error = error_get_last();
             error_log("Rename failed: Could not rename directory from " . $old_full . " to " . $new_full . " Error: " . ($error['message'] ?? 'Unknown error'));
             return false;
        }
    }

    error_log("Rename failed: Source item not found: " . $old_full . " or " . $old_full . ".json");
    return false;
}


function move_item($source_path, $target_folder_path) {
    $source_path = sanitize_path($source_path);
    $target_folder_path = ($target_folder_path === '') ? '' : sanitize_path($target_folder_path);

    if (empty($source_path)) {
        error_log("Move failed: Source path is empty.");
        return false;
    }

    $item_name = basename($source_path);
    $new_path = ($target_folder_path ? $target_folder_path . '/' : '') . $item_name;

     if ($source_path === $new_path || strpos($target_folder_path . '/', $source_path . '/') === 0 || $target_folder_path === $source_path) {
         error_log("Move failed: Cannot move item '$source_path' into itself ('$target_folder_path') or to the same location ('$new_path').");
         return false;
     }

    $old_full = CONTENT_DIR . $source_path;
    $new_full = CONTENT_DIR . $new_path;
    $target_dir_full = rtrim(CONTENT_DIR . $target_folder_path, '/');

    if (!is_dir($target_dir_full)) {
        if (!@mkdir($target_dir_full, 0755, true)) {
             $error = error_get_last();
             error_log("Move failed: Could not create target directory: " . $target_dir_full . " Error: " . ($error['message'] ?? 'Unknown error'));
             return false;
        }
    }

    if (file_exists($old_full . '.json')) {
        $old_full_file = $old_full . '.json';
        $new_full_file = $new_full . '.json';

        if (file_exists($new_full_file)) {
            error_log("Move failed: Destination page already exists: " . $new_full_file);
            return false;
        }

        $page_data = get_page_data($source_path);
        if ($page_data === false) {
             error_log("Move failed: Could not read source page data for: " . $source_path);
             return false;
        }
        $uuid_to_update = $page_data['uuid'] ?? null;
         if (!$uuid_to_update) {
             error_log("Move failed: Could not get UUID for page: " . $source_path);
         }

        if (!@rename($old_full_file, $new_full_file)) {
             error_log("Move failed: Could not move JSON file from " . $old_full_file . " to " . $new_full_file);
             return false;
        }

        $old_flat_slug = path_to_flat_slug($source_path);
        $new_flat_slug = path_to_flat_slug($new_path);
        $revision_files = glob(REVISIONS_DIR . $old_flat_slug . '-*.txt');
        foreach ($revision_files as $file) {
            $new_rev_name = str_replace($old_flat_slug, $new_flat_slug, basename($file));
            @rename($file, REVISIONS_DIR . $new_rev_name);
        }

        if ($uuid_to_update) {
            update_uuid_index($uuid_to_update, $new_path);
        }
        return true;

    }
    elseif (is_dir($old_full)) {
        if (file_exists($new_full)) {
             error_log("Move failed: Destination folder already exists: " . $new_full);
             return false;
        }

        if (@rename($old_full, $new_full)) {
            $index = load_uuid_index();
            $changed = false;
            $old_path_prefix = rtrim($source_path, '/') . '/';
            $new_path_prefix = rtrim($new_path, '/') . '/';

            foreach ($index as $uuid => $current_path) {
                if (strpos($current_path, $old_path_prefix) === 0) {
                     $updated_path = $new_path_prefix . substr($current_path, strlen($old_path_prefix));
                     $index[$uuid] = $updated_path;
                     $changed = true;
                }
                 else if ($current_path === $source_path) {
                    error_log("Warning during move: UUID index points directly to moved folder: " . $source_path);
                 }
            }
            if ($changed) {
                save_uuid_index($index);
            }
            return true;
        } else {
             $error = error_get_last();
             error_log("Move failed: Could not move directory from " . $old_full . " to " . $new_full . " Error: " . ($error['message'] ?? 'Unknown error'));
             return false;
        }
    } else {
         error_log("Move failed: Source item not found: " . $old_full . " or " . $old_full . ".json");
         return false;
    }
}


/**
 * Reverts a page to the content of an older revision.
 * This creates a NEW revision identical to the old one.
 * @param string $path The path of the page.
 * @param string $revision_id_to_revert The ID of the revision to revert TO.
 * @param string $username The username performing the revert action.
 * @return bool True on success, false on failure.
 */
function revert_to_revision($path, $revision_id_to_revert, $username) {
    $path = sanitize_path($path);
    $page_data = get_page_data($path);

    // Ensure history exists and is array before checking
    $history = $page_data['history'] ?? [];
    if (!is_array($history)) $history = [];

    // Basic checks - compare as strings
    if (!$page_data || empty($history) || !in_array((string)$revision_id_to_revert, array_map('strval',$history))) {
        error_log("Revert failed: Page data invalid or revision ID '$revision_id_to_revert' not found in history for '$path'.");
        return false;
    }

    $old_content = get_revision_content($path, $revision_id_to_revert);
    if (strpos($old_content, 'Error:') === 0) {
        error_log("Revert failed: Could not get content for revision ID '$revision_id_to_revert' for page '$path'. Error: $old_content");
        return false;
    }

    $title = $page_data['title'] ?? 'Untitled';
    $feature_image = $page_data['feature_image'] ?? '';
    $is_private = $page_data['is_private'] ?? false;

    // Call save_page_content - this handles creating the new revision file
    // and updating the page's JSON data (current_revision_id, history array)
    $success = save_page_content($path, $title, $old_content, $username, $feature_image, $is_private);

    if (!$success) {
         error_log("Revert failed: save_page_content failed when saving reverted content for page '$path'.");
    }

    return $success;
}


function search_pages($query) {
    $results = [];
    if (!is_dir(CONTENT_DIR)) return [];

    try {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator(CONTENT_DIR, FilesystemIterator::SKIP_DOTS | FilesystemIterator::UNIX_PATHS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $fileinfo) {
             if (!$fileinfo->isReadable()) continue;

            if ($fileinfo->isFile() && $fileinfo->getExtension() === 'json') {
                 $content = @file_get_contents($fileinfo->getPathname());
                 if ($content === false) continue;
                 $data = json_decode($content, true);
                 if (!is_array($data)) continue;

                $fullPath = $fileinfo->getPathname();
                 $relativePath = substr($fullPath, strlen(CONTENT_DIR));
                 $pagePath = substr($relativePath, 0, -5);

                $title = $data['title'] ?? basename($pagePath);

                // Skip private pages unless user can view them
                if (!empty($data['is_private']) && !can_view_private_page($data)) {
                    continue;
                }

                if (stripos($title, $query) !== false) {
                    $results[$pagePath] = [
                        'path' => $pagePath,
                        'title' => $title,
                        'context' => 'Match in title.'
                    ];
                    continue;
                }

                $currentRevisionId = $data['current_revision_id'] ?? null;
                 if ($currentRevisionId) {
                    $rev_content = get_revision_content($pagePath, $currentRevisionId);
                    if (strpos($rev_content, 'Error:') !== 0 && stripos($rev_content, $query) !== false) {
                        if (!isset($results[$pagePath])) {
                             $results[$pagePath] = [
                                'path' => $pagePath,
                                'title' => $title,
                                'context' => 'Match in content.'
                            ];
                        }
                    }
                }
            }
        }
    } catch (UnexpectedValueException $e) {
        error_log("Error during directory iteration in search: " . $e->getMessage());
    } catch (Exception $e) {
         error_log("General error during search: " . $e->getMessage());
    }

    return array_values($results);
}


// --- Content Retrieval for Lists ---

function get_all_pages_sorted_by_date() {
    $pages = [];
    if (!is_dir(CONTENT_DIR)) return [];

    try {
        $iterator = new RecursiveIteratorIterator(
             new RecursiveDirectoryIterator(CONTENT_DIR, FilesystemIterator::SKIP_DOTS | FilesystemIterator::UNIX_PATHS),
             RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $fileinfo) {
             if (!$fileinfo->isReadable()) continue;

            if ($fileinfo->isFile() && $fileinfo->getExtension() === 'json') {
                 $fullPath = $fileinfo->getPathname();
                 $relativePath = substr($fullPath, strlen(CONTENT_DIR));
                 $pagePath = substr($relativePath, 0, -5);

                $data = get_page_data($pagePath); // Handles UUID migration
                 if ($data) {
                     // Skip private pages unless user can view them
                     if (!empty($data['is_private']) && !can_view_private_page($data)) {
                         continue;
                     }
                     $pages[] = $data;
                 } else {
                      error_log("get_all_pages skipped invalid page data for path: " . $pagePath);
                 }
             }
         }
     } catch (UnexpectedValueException $e) {
         error_log("Error during directory iteration in get_all_pages: " . $e->getMessage());
     } catch (Exception $e) {
         error_log("General error during get_all_pages: " . $e->getMessage());
     }

    usort($pages, function($a, $b) {
        return ($b['current_revision_id'] ?? 0) <=> ($a['current_revision_id'] ?? 0);
    });

    return $pages;
}


function get_pages_by_author($username) {
    $all_pages = get_all_pages_sorted_by_date();
    $author_pages = [];
    foreach ($all_pages as $page) {
        if (($page['author'] ?? null) === $username) {
            $author_pages[] = $page;
        }
    }
    return $author_pages;
}


// --- Utility Functions ---

function load_template($template_name, $data = []) {
    global $config;
    extract($data);

    $header_path = __DIR__ . '/../../templates/partials/header.php';
    $content_path = __DIR__ . '/../../templates/' . $template_name . '.php';
    $footer_path = __DIR__ . '/../../templates/partials/footer.php';

    ob_start();
    $header_included = $content_included = $footer_included = false;

    if (file_exists($header_path)) {
         require $header_path;
         $header_included = true;
    } else {
         error_log("Header template not found: " . $header_path);
         echo "";
    }

    if (file_exists($content_path)) {
         require $content_path;
         $content_included = true;
    } else {
         error_log("Content template not found: " . $content_path);
         echo "";
         echo "<div class='window'><div class='window-content'><p style='color:red;'>Error: Could not load page content ($template_name).</p></div></div>";
    }

    if (file_exists($footer_path)) {
         require $footer_path;
         $footer_included = true;
    } else {
         error_log("Footer template not found: " . $footer_path);
         echo "";
    }

    ob_end_flush();
}


function load_template_rss($template_name, $data = []) {
    global $config;
    extract($data);

    if (!headers_sent()) {
        header("Content-Type: application/rss+xml; charset=utf-8");
    } else {
        error_log("Cannot set RSS header - headers already sent.");
    }

    $rss_template_path = __DIR__ . '/../../templates/' . $template_name . '.php';

    if (file_exists($rss_template_path)) {
        require $rss_template_path;
    } else {
        error_log("RSS template not found: " . $rss_template_path);
        echo '<?xml version="1.0" encoding="UTF-8" ?><rss version="2.0"><channel><title>Error</title><link>/</link><description>Feed template not found.</description></channel></rss>';
    }
    exit;
}


function e($string) {
    return htmlspecialchars((string)$string, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}


function slugify($text) {
    $text_backup_for_slugify = (string)$text; // Backup original
    if (mb_detect_encoding((string)$text, 'UTF-8', true) === false) {
        $text = utf8_encode((string)$text);
    }
    $text = @iconv('utf-8', 'us-ascii//IGNORE//TRANSLIT', (string)$text);

    if ($text === false) {
        $text = (string)$text_backup_for_slugify;
        $text = preg_replace('/[^a-zA-Z0-9\s-]/', '', $text);
    }

    $text = strtolower($text);
    $text = preg_replace('~[^a-z0-9]+~', '-', $text);
    $text = preg_replace('~-+~', '-', $text);
    $text = trim($text, '-');

    if (empty($text)) {
        return 'n-a';
    }

    // $text = substr($text, 0, 100); // Optional length limit

    return $text;
}

// --- Hashtag Functions ---

/**
 * Load hashtag index from storage
 * Structure: { "hashtag": { "count": 10, "pages": ["page_uuid1", "page_uuid2"] } }
 */
function load_hashtag_index() {
    if (!file_exists(HASHTAG_INDEX_FILE)) {
        return [];
    }
    $data = file_get_contents(HASHTAG_INDEX_FILE);
    return $data ? json_decode($data, true) : [];
}

/**
 * Save hashtag index to storage
 */
function save_hashtag_index($index) {
    $dir = dirname(HASHTAG_INDEX_FILE);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
    return file_put_contents(HASHTAG_INDEX_FILE, json_encode($index, JSON_PRETTY_PRINT)) !== false;
}

/**
 * Extract hashtags from content
 * Returns array of unique hashtags (lowercase, without #)
 * ONLY extracts hashtags from inside code blocks (inline `code` or fenced ```)
 */
function extract_hashtags($content) {
    if (empty($content)) {
        return [];
    }

    $hashtags = [];

    // 1. Extract hashtags from fenced code blocks (```...```)
    preg_match_all('/```[\s\S]*?```/u', $content, $fenced_blocks);
    foreach ($fenced_blocks[0] as $block) {
        // Match #tag (no space after #), can be at start or after any character
        preg_match_all('/#(?!\\s)([a-zA-Z0-9_-]+)/u', $block, $matches);
        if (!empty($matches[1])) {
            $hashtags = array_merge($hashtags, $matches[1]);
        }
    }

    // 2. Extract hashtags from inline code (`...`)
    // Use [^`\n\r] to prevent matching across newlines (which would interfere with fenced blocks)
    preg_match_all('/`[^`\n\r]+`/u', $content, $inline_blocks);
    foreach ($inline_blocks[0] as $block) {
        // Match #tag (no space after #), can be at start or after any character
        preg_match_all('/#(?!\\s)([a-zA-Z0-9_-]+)/u', $block, $matches);
        if (!empty($matches[1])) {
            $hashtags = array_merge($hashtags, $matches[1]);
        }
    }

    // 3. Extract hashtags from indented code blocks (4+ spaces at start of line)
    preg_match_all('/^[ ]{4,}.*/m', $content, $indented_blocks);
    foreach ($indented_blocks[0] as $block) {
        // Match #tag (no space after #), can be at start or after any character
        preg_match_all('/#(?!\\s)([a-zA-Z0-9_-]+)/u', $block, $matches);
        if (!empty($matches[1])) {
            $hashtags = array_merge($hashtags, $matches[1]);
        }
    }

    if (empty($hashtags)) {
        return [];
    }

    // Normalize to lowercase and remove duplicates
    $hashtags = array_map('strtolower', $hashtags);
    return array_unique($hashtags);
}

/**
 * Update hashtag index for a page
 * Removes old hashtags and adds new ones
 */
function update_page_hashtags($page_uuid, $new_hashtags) {
    $index = load_hashtag_index();

    // Remove this page from all hashtags first
    foreach ($index as $tag => $data) {
        if (isset($data['pages']) && in_array($page_uuid, $data['pages'])) {
            $index[$tag]['pages'] = array_values(array_diff($data['pages'], [$page_uuid]));
            $index[$tag]['count'] = count($index[$tag]['pages']);

            // Remove hashtag entry if no pages use it
            if ($index[$tag]['count'] === 0) {
                unset($index[$tag]);
            }
        }
    }

    // Add new hashtags
    foreach ($new_hashtags as $tag) {
        if (!isset($index[$tag])) {
            $index[$tag] = [
                'count' => 0,
                'pages' => []
            ];
        }

        if (!in_array($page_uuid, $index[$tag]['pages'])) {
            $index[$tag]['pages'][] = $page_uuid;
            $index[$tag]['count'] = count($index[$tag]['pages']);
        }
    }

    save_hashtag_index($index);
}

/**
 * Get pages by hashtag
 * Returns array of page data
 */
function get_pages_by_hashtag($hashtag) {
    $hashtag = strtolower(trim($hashtag));
    $index = load_hashtag_index();

    if (!isset($index[$hashtag]) || empty($index[$hashtag]['pages'])) {
        return [];
    }

    $uuid_index = load_uuid_index();
    $pages = [];

    foreach ($index[$hashtag]['pages'] as $page_uuid) {
        // Find path from UUID (UUID index is: uuid => path)
        $path = $uuid_index[$page_uuid] ?? null;
        if ($path !== null) {
            $page_data = get_page_data($path);
            if ($page_data && can_view_private_page($page_data)) {
                $pages[] = $page_data;
            }
        }
    }

    return $pages;
}

/**
 * Get top hashtags by usage count
 * Returns array of hashtags with their counts, sorted by count descending
 */
function get_top_hashtags($limit = 100) {
    $index = load_hashtag_index();

    // Sort by count descending
    uasort($index, function($a, $b) {
        return $b['count'] - $a['count'];
    });

    // Return top N
    $result = [];
    $count = 0;
    foreach ($index as $tag => $data) {
        if ($count >= $limit) {
            break;
        }
        $result[$tag] = $data['count'];
        $count++;
    }

    return $result;
}

/**
 * Search hashtags by prefix (for autocomplete)
 * Returns array of matching hashtag strings
 */
function search_hashtags($prefix, $limit = 10) {
    $prefix = strtolower(trim($prefix));
    if (empty($prefix)) {
        return [];
    }

    $index = load_hashtag_index();
    $matches = [];

    foreach ($index as $tag => $data) {
        if (strpos($tag, $prefix) === 0) {
            $matches[] = [
                'tag' => $tag,
                'count' => $data['count']
            ];
        }
    }

    // Sort by count descending
    usort($matches, function($a, $b) {
        return $b['count'] - $a['count'];
    });

    // Return top N tag names only
    return array_slice(array_column($matches, 'tag'), 0, $limit);
}