<?php
// app/core/auth.php

// Start session with longer lifetime for persistent sessions
ini_set('session.gc_maxlifetime', 30 * 24 * 60 * 60); // 30 days
session_set_cookie_params([
    'lifetime' => 30 * 24 * 60 * 60, // 30 days
    'path' => '/',
    'domain' => $_SERVER['HTTP_HOST'] ?? '',
    'secure' => isset($_SERVER['HTTPS']), // Use secure cookies if HTTPS
    'httponly' => true, // Prevent JavaScript access
    'samesite' => 'Lax' // CSRF protection
]);

// --- Storage Paths ---
// USERS_FILE is defined in functions.php
// Remember me tokens file - MUST be defined before session_start() and check_remember_me_cookie()
define('REMEMBER_TOKENS_FILE', __DIR__ . '/../storage/remember_tokens.json');

session_start();

// Check for remember me cookie and restore session
check_remember_me_cookie();

// --- User & Session Functions ---

/**
 * Load remember me tokens from storage
 */
function load_remember_tokens() {
    if (!file_exists(REMEMBER_TOKENS_FILE)) {
        return [];
    }
    $data = file_get_contents(REMEMBER_TOKENS_FILE);
    return $data ? json_decode($data, true) : [];
}

/**
 * Save remember me tokens to storage
 */
function save_remember_tokens($tokens) {
    $dir = dirname(REMEMBER_TOKENS_FILE);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
    return file_put_contents(REMEMBER_TOKENS_FILE, json_encode($tokens, JSON_PRETTY_PRINT)) !== false;
}

/**
 * Create a remember me token for a user
 */
function create_remember_token($username) {
    try {
        $selector = bin2hex(random_bytes(16));
        $token = bin2hex(random_bytes(32));
        $token_hash = hash('sha256', $token);

        $tokens = load_remember_tokens();
        $tokens[$selector] = [
            'username' => $username,
            'token_hash' => $token_hash,
            'expires' => time() + (30 * 24 * 60 * 60) // 30 days
        ];
        save_remember_tokens($tokens);

        // Set cookie with selector:token
        setcookie(
            'remember_me',
            $selector . ':' . $token,
            time() + (30 * 24 * 60 * 60), // 30 days
            '/',
            $_SERVER['HTTP_HOST'] ?? '',
            isset($_SERVER['HTTPS']),
            true // httponly
        );

        return true;
    } catch (Exception $e) {
        error_log("Failed to create remember token: " . $e->getMessage());
        return false;
    }
}

/**
 * Check and validate remember me cookie
 */
function check_remember_me_cookie() {
    // If already logged in, skip
    if (isset($_SESSION['user'])) {
        return;
    }

    // Check for remember me cookie
    if (!isset($_COOKIE['remember_me'])) {
        return;
    }

    $cookie_parts = explode(':', $_COOKIE['remember_me']);
    if (count($cookie_parts) !== 2) {
        delete_remember_cookie();
        return;
    }

    list($selector, $token) = $cookie_parts;
    $tokens = load_remember_tokens();

    if (!isset($tokens[$selector])) {
        delete_remember_cookie();
        return;
    }

    $token_data = $tokens[$selector];

    // Check if token has expired
    if ($token_data['expires'] < time()) {
        unset($tokens[$selector]);
        save_remember_tokens($tokens);
        delete_remember_cookie();
        return;
    }

    // Verify token
    $token_hash = hash('sha256', $token);
    if (!hash_equals($token_data['token_hash'], $token_hash)) {
        delete_remember_cookie();
        return;
    }

    // Valid token - restore session
    $_SESSION['user'] = $token_data['username'];

    // Load user role
    $users = load_users();
    if (isset($users[$token_data['username']])) {
        $_SESSION['role'] = $users[$token_data['username']]['role'] ?? 'editor';
    }

    session_regenerate_id(true);

    // Create new token for rotation
    unset($tokens[$selector]);
    save_remember_tokens($tokens);
    create_remember_token($token_data['username']);
}

/**
 * Delete remember me cookie
 */
function delete_remember_cookie() {
    setcookie(
        'remember_me',
        '',
        time() - 3600,
        '/',
        $_SERVER['HTTP_HOST'] ?? '',
        isset($_SERVER['HTTPS']),
        true
    );
}

/**
 * Checks login credentials against stored user data.
 * @param string $username
 * @param string $password
 * @param bool $remember Whether to create a remember me token
 * @return bool True on success, false on failure.
 */
function check_login($username, $password, $remember = false) {
    // Basic input validation
    if (empty($username) || empty($password)) {
        return false;
    }

    $users = load_users(); // Assumes load_users() is available (defined in functions.php)

    if (isset($users[$username])) {
        $user = $users[$username];
        // Verify the hashed password
        if (isset($user['password_hash']) && password_verify($password, $user['password_hash'])) {
            // Success! Set session variables
            $_SESSION['user'] = $user['username'];
            $_SESSION['role'] = $user['role'] ?? 'editor'; // Default role if missing
            session_regenerate_id(true); // Security: Prevent session fixation

            // Create remember me token if requested
            if ($remember) {
                create_remember_token($username);
            }

            return true;
        }
    }
    // Log failed login attempt (optional, consider rate limiting)
    // error_log("Failed login attempt for user: " . $username);
    return false; // Failed login
}

/**
 * Logs the user out by destroying the session.
 */
function logout() {
    // Remove remember me token if exists
    if (isset($_COOKIE['remember_me'])) {
        $cookie_parts = explode(':', $_COOKIE['remember_me']);
        if (count($cookie_parts) === 2) {
            $selector = $cookie_parts[0];
            $tokens = load_remember_tokens();
            if (isset($tokens[$selector])) {
                unset($tokens[$selector]);
                save_remember_tokens($tokens);
            }
        }
        delete_remember_cookie();
    }

    // Unset all session variables
    $_SESSION = array();

    // If using session cookies, delete the cookie
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }

    // Finally, destroy the session.
    session_destroy();
}

/**
 * Checks if the current user is logged in as an admin.
 * @return bool
 */
function is_admin() {
    return isset($_SESSION['role']) && $_SESSION['role'] === 'admin';
}

/**
 * Checks if the current user is logged in as an editor or admin.
 * @return bool
 */
function is_editor() {
    $role = $_SESSION['role'] ?? 'guest';
    return $role === 'admin' || $role === 'editor';
}

/**
 * Checks if any user is currently logged in.
 * @return bool
 */
function is_logged_in() {
    return isset($_SESSION['user']) && !empty($_SESSION['user']);
}

/**
 * Gets the current logged-in username.
 * @return string|null
 */
function get_current_username() {
    return $_SESSION['user'] ?? null;
}

/**
 * Redirects to login if the user doesn't have the required role.
 * Use this at the top of protected pages/actions.
 * @param string $role 'editor' or 'admin' (minimum required role)
 */
function require_auth($role = 'editor') {
    $logged_in = isset($_SESSION['user']);
    $user_role = $_SESSION['role'] ?? 'guest';

    $authorized = false;
    if ($logged_in) {
        if ($role === 'admin' && $user_role === 'admin') {
            $authorized = true;
        } elseif ($role === 'editor' && ($user_role === 'admin' || $user_role === 'editor')) {
            $authorized = true;
        }
    }

    if (!$authorized) {
        // Optional: Store intended destination to redirect back after login
        // $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
        header('Location: /login');
        exit;
    }
}

// --- CSRF (Cross-Site Request Forgery) Protection ---

/**
 * Generates a CSRF token and stores it in the session if not already set.
 * @return string The CSRF token.
 */
function generate_csrf_token() {
    if (empty($_SESSION['csrf_token'])) {
        try {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        } catch (Exception $e) {
            // Handle error if random_bytes fails
            error_log("Failed to generate CSRF token: " . $e->getMessage());
            // Fallback or die
            $_SESSION['csrf_token'] = 'fallback_token_' . time(); // Less secure fallback
        }
    }
    return $_SESSION['csrf_token'];
}

/**
 * Validates a submitted CSRF token against the one in the session.
 * Dies with an error message if validation fails.
 * @param string $token The submitted token.
 */
function validate_csrf_token($token) {
    // Check if token exists in session and matches submitted token securely
    if (!isset($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)) {
        // CSRF token mismatch - log potentially malicious attempt
        error_log("CSRF validation failed. Submitted token: '$token', Session token: '" . ($_SESSION['csrf_token'] ?? 'Not Set') . "'");
        // Provide a generic error message to the user
        die('CSRF validation failed. Please try submitting the form again.');
    }
    // Token is valid. Do NOT unset it here to allow multiple forms/back button usage.
}

// --- User Management (Admin Actions - Called from index.php) ---

/**
 * Adds a new user (admin action).
 * @param string $username
 * @param string $password
 * @param string $role 'admin' or 'editor'
 * @param string $invited_by Username of the admin adding the user.
 * @return bool True on success, false on failure.
 */
function add_user($username, $password, $role, $invited_by) {
    // Basic validation
    if (empty($username) || empty($password) || empty($role)) {
        error_log("Add user failed: Empty username, password, or role.");
        return false;
    }
    // Validate username format (e.g., alphanumeric) if desired
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $username)) {
        error_log("Add user failed: Invalid username format for '$username'.");
        return false;
    }


    $users = load_users(); // Assumes load_users() is available (defined in functions.php)
    if (isset($users[$username])) {
        error_log("Add user failed: Username '$username' already exists.");
        return false; // User already exists
    }

    // Hash the password
    $hash_options = [];
    $password_hash = password_hash($password, PASSWORD_ARGON2ID, $hash_options);
     if ($password_hash === false) {
         error_log("Password hashing failed for new user: " . $username);
         return false;
     }

    // Add the new user data
    $users[$username] = [
        'username' => $username,
        'password_hash' => $password_hash,
        'role' => ($role === 'admin') ? 'admin' : 'editor', // Ensure valid role
        'invited_by' => $invited_by,
        // Initialize profile fields (optional)
        'tagline' => '',
        'photo_url' => '',
        'bio' => ''
    ];

    // Save the updated user list
    return save_users($users); // Assumes save_users() is available (defined in functions.php)
}

/**
 * Deletes a user (admin action).
 * Prevents deleting the last admin or the 'admin' user if only one exists.
 * @param string $username
 * @return bool True on success, false on failure.
 */
function delete_user($username) {
    // Basic validation
    if (empty($username)) {
        return false;
    }

    $users = load_users();

    // Check if user exists
    if (!isset($users[$username])) {
        error_log("Delete user failed: User '$username' not found.");
        return false;
    }

    // Prevent deleting the 'admin' user if it's the only admin left
    $admin_count = 0;
    foreach ($users as $user_data) {
        if (($user_data['role'] ?? 'editor') === 'admin') {
            $admin_count++;
        }
    }

    if (($users[$username]['role'] ?? 'editor') === 'admin' && $admin_count <= 1) {
        error_log("Delete user failed: Cannot delete the last remaining admin ('$username').");
        return false; // Can't delete the last admin
    }

    // Remove the user
    unset($users[$username]);

    // Save the updated user list
    return save_users($users);
}


// --- Update User Password (User Action - Called from index.php) ---

/**
 * Updates a user's password securely (user action).
 * @param string $username The user whose password to change.
 * @param string $new_password The new plain-text password.
 * @return bool True on success, false on failure.
 */
function update_user_password($username, $new_password) {
    // Basic validation
    if (empty($username) || empty($new_password)) {
        error_log("Update password failed: Empty username or password for '$username'.");
        return false;
    }
     // Optional: Add password strength requirements check here

    $users = load_users();

    if (!isset($users[$username])) {
        error_log("Update password failed: User '$username' not found.");
        return false; // User doesn't exist
    }

    // Hash the new password
    $hash_options = [];
    $new_hash = password_hash($new_password, PASSWORD_ARGON2ID, $hash_options);

    if ($new_hash === false) {
         error_log("Password hashing failed during update for user: " . $username);
         return false;
    }

    // Update the user's data
    $users[$username]['password_hash'] = $new_hash;

    // Save the updated user data
    return save_users($users);
}