# BookGram Plugin System - User Guide

## Overview

The BookGram plugin system allows you to embed interactive plugins directly into your articles using simple shortcode syntax. Each article can have its own isolated plugin instances with separate data storage.

## ✅ What Has Been Implemented

### Core Features
- **Plugin Discovery**: Automatically discovers plugins in the root directory with "plugin_" prefix
- **Shortcode Embedding**: Use `[plugin:name id=instance]` syntax to embed plugins in articles
- **Per-Article Instances**: Each article gets isolated plugin instances with separate data
- **Admin Session Passthrough**: BookGram admin credentials automatically work in plugin admin panels
- **CSS Override**: Plugins are wrapped with BookGram's CSS theme for consistent styling
- **Plugin Manager UI**: Built into the article editor for easy plugin insertion
- **Instance Cleanup**: Plugin instances are automatically deleted when articles are deleted

### Existing Plugins
Three plugins are now integrated:
1. **QuizGram** - Interactive quiz system
2. **JoinGram** - Form builder and survey system
3. **QuesChan** - Discussion board system

## How to Use

### 1. Creating an Article with Plugins

1. Navigate to the article editor (create or edit an article)
2. Scroll down to the **Plugin Manager** section below the content textarea
3. Select a plugin from the dropdown menu
4. Enter a unique Instance ID (e.g., "quiz-intro", "form-signup", "discussion-main")
5. Click "Insert Shortcode"
6. Save the article

**Example Shortcodes:**
```
[plugin:quizgram id=intro-quiz]

[plugin:joingram id=newsletter-signup]

[plugin:queschan id=community-discussion]
```

### 2. Configuring Plugin Instances

After inserting a shortcode and saving your article:

1. The Plugin Manager will show all existing instances in the article
2. Click the **[Configure]** link next to any instance
3. This opens the plugin's admin panel with your BookGram credentials automatically authenticated
4. Configure the plugin (add quiz questions, form fields, board settings, etc.)

### 3. Viewing Embedded Plugins

When you view the article, the plugin will be embedded inline:
- If not configured yet, it shows a "Configure" message
- If configured, it displays the plugin interface (quiz start button, form link, discussion board, etc.)

## File Structure

### Core Plugin System Files

```
bookgram/v10.00/
├── app/
│   ├── core/
│   │   ├── plugins.php              # Core plugin system
│   │   └── plugin_auth.php          # Authentication helpers for plugins
│   └── storage/
│       ├── plugin_instances/        # Plugin instance data (per article)
│       │   └── {page_uuid}/
│       │       └── {plugin_id}/
│       │           └── {instance_id}/
│       │               ├── instance.json
│       │               └── data/
│       ├── plugin_registry.json     # Plugin cache
│       └── plugin_auth_tokens.json  # Auth tokens
├── public/
│   └── css/
│       └── plugin-wrapper.css       # Plugin CSS isolation
└── plugin_{name}/                   # Plugin directories
    ├── plugin.json                  # Plugin manifest
    ├── embed.php                    # Embed handler
    ├── setup.php                    # Instance initialization
    └── admin/                       # Admin panel
```

### Modified BookGram Files

- `index.php` - Added `/plugin-admin/` route for admin passthrough
- `templates/page.php` - Added shortcode parsing
- `templates/edit.php` - Added Plugin Manager UI
- `templates/partials/header.php` - Load plugin wrapper CSS
- `app/core/functions.php` - Added instance cleanup in `delete_page()`

## Plugin Manifest Schema

Each plugin requires a `plugin.json` file:

```json
{
  "id": "plugin_quizgram",
  "name": "QuizGram",
  "version": "1.0.0",
  "description": "Interactive quiz system",
  "supports_instances": true,
  "entry_point": "index.php",
  "admin_entry": "admin/index.php",
  "setup_script": "setup.php",
  "css_files": ["css/style.css"],
  "embed_handler": "embed.php",
  "required_permissions": ["editor"]
}
```

## Plugin Embed Handler

Each plugin needs an `embed.php` file that receives instance context via environment variables:

```php
<?php
// Get instance context
$instance_path = $_ENV['BOOKGRAM_INSTANCE_PATH'];
$instance_id = $_ENV['BOOKGRAM_INSTANCE_ID'];
$page_uuid = $_ENV['BOOKGRAM_PAGE_UUID'];
$params = json_decode($_ENV['BOOKGRAM_PARAMS'], true);

// Load instance data
$data_file = $instance_path . '/data/mydata.json';

// Display plugin content
echo '<div class="my-plugin">';
echo '<h3>Plugin Content Here</h3>';
echo '</div>';
```

## Admin Authentication

Plugins can authenticate against BookGram by including the plugin auth helper:

```php
<?php
// In plugin admin pages
require_once '../../app/core/plugin_auth.php';

// Verify authentication
$auth = bookgram_auth_check();
bookgram_require_auth();

// Get current user info
$username = bookgram_get_username();
$instance_id = bookgram_get_instance_id();
$instance_path = bookgram_get_instance_path();

// Check permissions
if (bookgram_is_admin()) {
    // Admin-only features
}
```

## Security Features

1. **Instance Isolation**: Each article's plugin instances have separate data directories
2. **Auth Tokens**: 1-hour expiration, cryptographically secure, tied to specific instances
3. **Input Validation**: Plugin IDs and instance IDs are sanitized
4. **Permission Checks**: Only editors/admins can configure plugins
5. **Path Traversal Prevention**: All paths use absolute references

## Testing

A test script is included to verify the plugin system:

```bash
php test_plugins.php
```

This will check:
- Plugin discovery
- Registry creation
- Storage directories
- Shortcode parsing
- Instance path generation

## Advanced Features (Future Enhancements)

The following features are planned but not yet implemented:

### Privacy Controls for Plugin Instances
Each instance could have:
- Private link access (share token)
- Password protection
- Access control per instance

To implement this, you would:
1. Create `templates/plugin-instance-settings.php`
2. Add a route handler in `index.php` for `/plugin-instance-settings/`
3. Update embed handlers to check privacy settings before rendering

### Plugin Hooks System
Allow plugins to hook into BookGram events:
- `on_article_save`
- `on_article_delete`
- `on_user_login`

### Plugin API Endpoints
Create RESTful endpoints for plugins:
- `/api/plugin/{plugin_id}/{instance_id}/data`
- `/api/plugin/{plugin_id}/{instance_id}/action`

## Troubleshooting

### Plugins Not Showing in Editor
1. Check that plugin folders start with "plugin_"
2. Verify `plugin.json` exists and is valid JSON
3. Run `php test_plugins.php` to check discovery

### Shortcode Not Rendering
1. Check shortcode syntax: `[plugin:name id=instance]`
2. Verify the plugin has an `embed.php` file
3. Check browser console for JavaScript errors

### Admin Panel Not Accessible
1. Ensure you're logged in as editor or admin
2. Check that the plugin has an `admin_entry` in `plugin.json`
3. Verify the auth token is being created (check `plugin_auth_tokens.json`)

### Instance Data Not Saving
1. Check directory permissions on `app/storage/plugin_instances/`
2. Verify the instance was initialized (check for `instance.json`)
3. Look for errors in PHP error log

## Support

For issues or questions:
1. Check the test script output: `php test_plugins.php`
2. Review PHP error logs
3. Verify file permissions on storage directories
4. Check browser console for JavaScript errors

## Summary

The plugin system is now fully functional and ready to use! You can:
- ✅ Discover plugins automatically
- ✅ Insert plugins into articles via shortcodes
- ✅ Manage plugin instances per article
- ✅ Configure plugins using BookGram admin credentials
- ✅ Have isolated data storage per instance
- ✅ Automatically clean up when articles are deleted

Enjoy your new plugin system!
