# Plugin Admin Modal System

## Overview

All plugin configuration interfaces now open in OS/2-themed modal dialog boxes instead of navigating to separate pages. This creates a more integrated, desktop-like experience that matches the BookGram aesthetic.

## Features

### OS/2 Warp Theme
- **Window Controls**: Minimize, maximize, and close buttons styled like classic OS/2
- **Title Bar**: Blue gradient with white text (classic OS/2 active window style)
- **Draggable**: Click and drag the title bar to reposition the modal
- **Resizable**: Maximize button toggles between windowed and full-screen
- **Shadow**: 3D raised border effect with shadow

### User Experience
- **No Page Navigation**: Configuration opens in-place as popup
- **Quick Access**: Click "Configure" links to open admin panel instantly
- **Auto-Close**: Modal closes automatically after successful save
- **Page Refresh**: Parent page reloads to show updated plugin after closing
- **Keyboard Support**: Press `Escape` to close modal

### Visual Polish
- **Loading Indicator**: Shows "Loading plugin configuration..." with animated dots
- **Smooth Animations**: Fade-in overlay and slide-down modal entrance
- **iframe Integration**: Plugin admin pages load seamlessly inside modal
- **Background Blur**: Slight backdrop blur effect on overlay

## Technical Implementation

### Files Created

1. **`/public/js/plugin-admin-modal.js`**
   - Modal creation and management
   - Drag-and-drop functionality
   - Window controls (minimize/maximize/close)
   - Message passing between iframe and parent

2. **`/public/css/plugin-admin-modal.css`**
   - OS/2 Warp styling
   - Modal animations
   - Responsive design
   - Print-friendly (hides modal when printing)

### Files Modified

#### Header Integration
- **`templates/partials/header.php`**
  - Added CSS link: `/public/css/plugin-admin-modal.css`
  - Added JS script: `/public/js/plugin-admin-modal.js`

#### Plugin Embed Files (Configuration Links)
- **`plugin_PlayList/embed.php`**
- **`plugin_socialgram/embed.php`**
- **`plugin_joingram/embed.php`**

Changed from:
```php
<a href="/plugin-admin/{uuid}/{plugin}/{instance}">Configure</a>
```

To:
```php
<a href="#" onclick="event.preventDefault();
   openPluginAdmin('{uuid}', '{plugin}', '{instance}', 'PluginName');">
   Configure</a>
```

#### Plugin Admin Pages (Modal Detection)
- **`plugin_PlayList/admin/index.php`**
- **`plugin_socialgram/admin/index.php`**
- **`plugin_joingram/admin/dashboard.php`**
- **`plugin_joingram/admin/form-builder.php`**
- **`plugin_joingram/admin/responses.php`**

Added JavaScript at end of each file:
```javascript
// Modal mode detection and parent notification
(function() {
    const urlParams = new URLSearchParams(window.location.search);
    const isModal = urlParams.get('modal') === '1';

    if (isModal) {
        // Hide "Return to BookGram" link
        const returnLink = document.querySelector('a[href="/"]');
        if (returnLink && returnLink.textContent.includes('Return to BookGram')) {
            returnLink.style.display = 'none';
        }

        // Notify parent on successful save
        <?php if ($success_message): ?>
        setTimeout(function() {
            if (window.parent && window.parent !== window) {
                window.parent.postMessage('plugin-admin-saved', '*');
            }
        }, 1500);
        <?php endif; ?>

        // Add modal styling adjustments
        document.body.style.margin = '0';
        document.body.style.padding = '10px';
        document.body.style.background = '#008080';
    }
})();
```

## Usage

### Opening Plugin Admin Modal

Use the global JavaScript function:
```javascript
openPluginAdmin(pageUuid, pluginId, instanceId, pluginName);
```

**Parameters:**
- `pageUuid`: UUID of the article containing the plugin
- `pluginId`: Plugin identifier (e.g., 'plugin_PlayList')
- `instanceId`: Unique instance ID
- `pluginName`: Display name for modal title (e.g., 'PlayList')

**Example:**
```javascript
openPluginAdmin('abc123', 'plugin_PlayList', 'myplaylist', 'PlayList');
```

### Closing Modal Programmatically

```javascript
closePluginAdmin();
```

Or from within an iframe:
```javascript
window.parent.postMessage('plugin-admin-close', '*');
```

### From Plugin Embed Files

```php
<a href="#"
   onclick="event.preventDefault();
            openPluginAdmin('<?php echo $page_uuid; ?>',
                          'plugin_name',
                          '<?php echo $instance_id; ?>',
                          'Display Name');">
   Configure Plugin
</a>
```

## URL Parameter

When modal opens, it appends `?modal=1` to the admin page URL:
```
/plugin-admin/{uuid}/{plugin}/{instance}?modal=1
```

Plugin admin pages detect this parameter and:
1. Hide "Return to BookGram" links
2. Adjust styling for iframe context
3. Send postMessage to parent on save success

## Message Passing

### From iframe to parent (on save success):
```javascript
window.parent.postMessage('plugin-admin-saved', '*');
```

### From iframe to parent (to close):
```javascript
window.parent.postMessage('plugin-admin-close', '*');
```

### Parent listens for messages:
```javascript
window.addEventListener('message', function(e) {
    if (e.data === 'plugin-admin-saved' || e.data === 'plugin-admin-close') {
        closeModal();
        setTimeout(() => window.location.reload(), 500);
    }
});
```

## Window Controls

### Minimize Button
- Hides modal briefly then shows it again (visual feedback)
- Does not minimize to taskbar (not implemented)

### Maximize Button
- First click: Expands to 95vw × 90vh
- Second click: Restores to original size (800px × 600px)
- Icon changes: `□` ↔ `❐`

### Close Button
- Red color on hover
- Closes modal and clears iframe
- Triggers page reload after delay

## Styling Details

### Modal Window
- Width: 90% (max 900px)
- Height: 80vh (max 700px)
- Position: Centered (absolute positioning)
- Border: 2px outset white (3D raised effect)
- Shadow: 4px × 4px black shadow
- Background: #c0c0c0 (silver/gray)

### Title Bar
- Background: Blue gradient (#000080 to #1084d0)
- Color: White with text shadow
- Height: 28px
- Cursor: Grab (changes to grabbing when dragging)

### Overlay
- Background: rgba(0, 128, 128, 0.3) - Semi-transparent teal
- Backdrop filter: 2px blur
- Blocks interaction with background

### Loading State
- White box with outset border
- Centered positioning
- Animated dots (...) using CSS animation
- Z-index 10 (above iframe)

## Browser Compatibility

- **Chrome/Edge**: Full support
- **Firefox**: Full support
- **Safari**: Full support (some backdrop-filter limitations)
- **Mobile**: Supported, dragging disabled on mobile

## Accessibility

- **Keyboard**: ESC key closes modal
- **Focus**: Modal traps focus when open
- **Screen readers**: Proper ARIA labels (can be enhanced)

## Responsive Behavior

### Desktop (> 768px)
- Standard sizing
- Draggable title bar
- Maximize/restore functionality

### Mobile (≤ 768px)
- Width: 95%
- Height: 90vh
- Dragging disabled
- Touch-friendly controls

## Performance

- **Lazy Loading**: Modal HTML created on first use only
- **iframe Cleanup**: Iframe src cleared when modal closes
- **Animation**: GPU-accelerated CSS transforms
- **Memory**: Reuses single modal instance

## Security

- **postMessage**: Uses origin checking (can be enhanced)
- **iframe**: Same-origin (plugin admin is part of BookGram)
- **XSS Protection**: All plugin names properly escaped

## Future Enhancements

Possible improvements:
- [ ] Remember last position/size per plugin
- [ ] Multiple modal support (stack modals)
- [ ] Minimize to launchpad (like Windows taskbar)
- [ ] Confirmation dialog on close if unsaved changes
- [ ] Better mobile experience (full-screen on small devices)
- [ ] Drag-to-resize corners
- [ ] Snap-to-edges functionality

## Testing

To test the modal system:

1. **Create a plugin instance**: Add `[plugin:name id=test]` to an article
2. **Click configure link**: Modal should open
3. **Test dragging**: Drag title bar to move window
4. **Test maximize**: Click maximize button, should expand
5. **Test close**: Click X, should close and reload page
6. **Save configuration**: Should auto-close after 1.5 seconds

## Troubleshooting

### Modal doesn't open
- Check browser console for JavaScript errors
- Verify CSS/JS files are loaded
- Check that `openPluginAdmin()` function exists

### iframe shows blank
- Check that plugin admin page loads directly at URL
- Verify `?modal=1` parameter is appended
- Check browser console in iframe for errors

### Modal doesn't close after save
- Verify `$success_message` is set in PHP
- Check that postMessage is being sent
- Verify parent listener is active

### Styling looks broken
- Check that `/public/css/plugin-admin-modal.css` is loaded
- Verify OS/2 theme CSS is not conflicting
- Check browser's developer tools for CSS errors

## Example: Complete Integration

### 1. Plugin Embed Link
```php
// In plugin_example/embed.php
if (!file_exists($config_file)) {
    echo '<p><a href="#"
          onclick="event.preventDefault();
                   openPluginAdmin(\'' . $page_uuid . '\',
                                 \'plugin_example\',
                                 \'' . $instance_id . '\',
                                 \'Example Plugin\');">
          Configure Plugin
         </a></p>';
    return;
}
```

### 2. Plugin Admin Page
```php
// In plugin_example/admin/index.php (at the end)
<script>
(function() {
    const urlParams = new URLSearchParams(window.location.search);
    const isModal = urlParams.get('modal') === '1';

    if (isModal) {
        document.querySelector('a[href="/"]')?.style.display = 'none';

        <?php if ($success_message): ?>
        setTimeout(() => {
            window.parent?.postMessage('plugin-admin-saved', '*');
        }, 1500);
        <?php endif; ?>

        document.body.style.background = '#008080';
    }
})();
</script>
```

### 3. Result
- User clicks "Configure Plugin"
- Modal opens with OS/2 styling
- User edits configuration
- Clicks "Save"
- Success message shows for 1.5 seconds
- Modal closes automatically
- Page reloads to show updated plugin

## Conclusion

The plugin admin modal system provides a polished, integrated experience that:
- Eliminates page navigation
- Maintains BookGram's OS/2 aesthetic
- Improves workflow efficiency
- Feels like a true desktop application

All plugins now benefit from this unified configuration interface.
